import sys
import os
import shutil
import sqlite3
import calendar
import subprocess
import platform
from datetime import date, datetime, timedelta
import gi
import holidays

# PDF Imports
try:
    from reportlab.lib import colors
    from reportlab.lib.pagesizes import A4, landscape
    from reportlab.pdfgen import canvas
    from reportlab.lib.units import cm
except ImportError:
    print("Bitte installiere reportlab: pip install reportlab")
    sys.exit(1)

gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, Gio, GObject, Pango, Gdk, GLib

# --- HILFSFUNKTIONEN ---

def get_documents_dir():
    """Findet den Dokumente-Ordner des Users."""
    docs_dir = GLib.get_user_special_dir(GLib.UserDirectory.DIRECTORY_DOCUMENTS)
    if not docs_dir: 
        docs_dir = os.path.expanduser("~/Documents")
    
    base_dir = os.path.join(docs_dir, "Arbeitszeiterfassung")
    if not os.path.exists(base_dir):
        os.makedirs(base_dir)
    return base_dir

def parse_time(t_str):
    if not t_str: return 0.0
    try:
        parts = t_str.split(':')
        return int(parts[0]) + (int(parts[1]) if len(parts) > 1 else 0)/60.0
    except: return 0.0

def compute_work_hours(start_str, end_str, pause_str):
    s = parse_time(start_str)
    e = parse_time(end_str)
    p = parse_time(pause_str)
    if s == 0 and e == 0: return 0.0
    if e < s: raw_work = (24.0 - s) + e
    else: raw_work = e - s
    return max(0.0, raw_work - p)

# --- DATENBANK ---

class Database:
    def __init__(self):
        # Datenpfad im User-Home (~/.local/share/arbeitszeit)
        data_dir = GLib.get_user_data_dir() 
        self.app_data_dir = os.path.join(data_dir, "arbeitszeit")
        
        if not os.path.exists(self.app_data_dir):
            os.makedirs(self.app_data_dir)
            
        self.db_name = os.path.join(self.app_data_dir, "arbeitszeit_v4.db")
        
        self.conn = sqlite3.connect(self.db_name)
        self.cursor = self.conn.cursor()
        self.create_tables()

    def create_tables(self):
        self.cursor.execute("""
            CREATE TABLE IF NOT EXISTS employees (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                first_name TEXT,
                last_name TEXT,
                address TEXT,
                phone TEXT,
                email TEXT,
                bundesland TEXT DEFAULT 'BY',
                vacation_days REAL DEFAULT 30.0,
                vacation_carryover REAL DEFAULT 0.0,
                target_0 REAL DEFAULT 8.0,
                target_1 REAL DEFAULT 8.0,
                target_2 REAL DEFAULT 8.0,
                target_3 REAL DEFAULT 8.0,
                target_4 REAL DEFAULT 8.0,
                target_5 REAL DEFAULT 0.0,
                target_6 REAL DEFAULT 0.0
            )
        """)
        self.cursor.execute("""
            CREATE TABLE IF NOT EXISTS entries (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                employee_id INTEGER,
                date TEXT,
                start_time TEXT,
                end_time TEXT,
                break_time TEXT,
                code TEXT,
                UNIQUE(employee_id, date)
            )
        """)
        self.conn.commit()

    def add_employee(self, f_name, l_name, addr, phone, email, state, vac_days, vac_over, targets):
        self.cursor.execute("""
            INSERT INTO employees 
            (first_name, last_name, address, phone, email, bundesland, vacation_days, vacation_carryover,
             target_0, target_1, target_2, target_3, target_4, target_5, target_6) 
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
        """, (f_name, l_name, addr, phone, email, state, vac_days, vac_over, *targets))
        self.conn.commit()

    def update_employee(self, emp_id, f_name, l_name, addr, phone, email, state, vac_days, vac_over, targets):
        self.cursor.execute("""
            UPDATE employees SET 
            first_name=?, last_name=?, address=?, phone=?, email=?, bundesland=?, 
            vacation_days=?, vacation_carryover=?,
            target_0=?, target_1=?, target_2=?, target_3=?, target_4=?, target_5=?, target_6=?
            WHERE id=?
        """, (f_name, l_name, addr, phone, email, state, vac_days, vac_over, *targets, emp_id))
        self.conn.commit()

    def delete_employee(self, emp_id):
        self.cursor.execute("DELETE FROM entries WHERE employee_id=?", (emp_id,))
        self.cursor.execute("DELETE FROM employees WHERE id=?", (emp_id,))
        self.conn.commit()

    def get_employees(self):
        self.cursor.execute("SELECT * FROM employees")
        return self.cursor.fetchall()
    
    def get_employee_by_id(self, emp_id):
        self.cursor.execute("SELECT * FROM employees WHERE id=?", (emp_id,))
        return self.cursor.fetchone()

    def get_entry(self, emp_id, date_str):
        self.cursor.execute("SELECT * FROM entries WHERE employee_id=? AND date=?", (emp_id, date_str))
        return self.cursor.fetchone()

    def upsert_entry(self, emp_id, date_str, start, end, pause, code):
        self.cursor.execute("""
            INSERT INTO entries (employee_id, date, start_time, end_time, break_time, code)
            VALUES (?, ?, ?, ?, ?, ?)
            ON CONFLICT(employee_id, date) DO UPDATE SET
            start_time=excluded.start_time,
            end_time=excluded.end_time,
            break_time=excluded.break_time,
            code=excluded.code
        """, (emp_id, date_str, start, end, pause, code))
        self.conn.commit()

    def get_taken_vacation_days(self, emp_id, year):
        search_str = f"{year}-%"
        self.cursor.execute("SELECT COUNT(*) FROM entries WHERE employee_id=? AND code='Urlaub' AND date LIKE ?", (emp_id, search_str))
        result = self.cursor.fetchone()
        return result[0] if result else 0

    def calculate_balance(self, emp_id, until_date_str=None):
        emp = self.get_employee_by_id(emp_id)
        if not emp: return 0.0
        state = emp[6]; targets = emp[9:16]
        
        query = "SELECT * FROM entries WHERE employee_id=?"
        params = [emp_id]
        if until_date_str:
            query += " AND date < ?"
            params.append(until_date_str)
            
        self.cursor.execute(query, tuple(params))
        entries = self.cursor.fetchall()
        total_balance = 0.0
        
        for entry in entries:
            d_str = entry[2]
            d_obj = date.fromisoformat(d_str)
            actual = compute_work_hours(entry[3], entry[4], entry[5])
            code = entry[6]
            target = targets[d_obj.weekday()]
            
            de_holidays = holidays.DE(subdiv=state, years=d_obj.year)
            if (d_obj in de_holidays) and target > 0: target = 0.0
            
            if (code in ["Urlaub", "Krank"]) and target > 0: diff = 0.0
            else: diff = actual - target
            total_balance += diff
        return total_balance

    def get_monthly_balances_for_year(self, emp_id, year):
        monthly_diffs = [0.0] * 12
        emp = self.get_employee_by_id(emp_id)
        if not emp: return monthly_diffs
        state = emp[6]; targets = emp[9:16]
        search_str = f"{year}-%"
        self.cursor.execute("SELECT * FROM entries WHERE employee_id=? AND date LIKE ?", (emp_id, search_str))
        entries = self.cursor.fetchall()
        for entry in entries:
            d_str = entry[2]
            d_obj = date.fromisoformat(d_str)
            idx = d_obj.month - 1
            actual = compute_work_hours(entry[3], entry[4], entry[5])
            code = entry[6]
            target = targets[d_obj.weekday()]
            if (d_obj in holidays.DE(subdiv=state, years=year)) and target > 0: target = 0.0
            if (code in ["Urlaub", "Krank"]) and target > 0: diff = 0.0
            else: diff = actual - target
            monthly_diffs[idx] += diff
        return monthly_diffs

    # --- NEUE BACKUP METHODE MIT ZIEL-ORDNER ---
    def create_backup(self, target_folder):
        timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
        dest = os.path.join(target_folder, f"backup_{timestamp}.db")
        shutil.copy2(self.db_name, dest)
        return dest

db = Database()

# --- PDF EXPORTS ---

class MonthlyPDF:
    def __init__(self, emp_data, year, month, vac_stats, prev_balance):
        self.emp_data = emp_data; self.year = year; self.month = month
        self.emp_id = emp_data[0]; self.targets = emp_data[9:16]; self.state = emp_data[6]
        self.vac_stats = vac_stats; self.prev_balance = prev_balance
        
        base_dir = get_documents_dir()
        self.filename = os.path.join(base_dir, f"Abrechnung_{emp_data[2]}_{year}_{month:02d}.pdf")
        
    def generate(self):
        c = canvas.Canvas(self.filename, pagesize=A4)
        width, height = A4
        
        c.setFont("Helvetica-Bold", 16)
        c.drawString(2*cm, height - 2*cm, "Monatsabrechnung Arbeitszeit")
        c.setFont("Helvetica", 10)
        c.drawString(2*cm, height - 3*cm, f"Mitarbeiter: {self.emp_data[1]} {self.emp_data[2]}")
        c.drawString(2*cm, height - 3.5*cm, f"Monat: {calendar.month_name[self.month]} {self.year}")
        c.drawString(10*cm, height - 3*cm, f"Personal-Nr: {self.emp_id:04d}")
        c.drawString(10*cm, height - 3.5*cm, f"Bundesland: {self.state}")

        c.setStrokeColor(colors.grey)
        c.rect(14*cm, height - 4*cm, 5*cm, 1.8*cm)
        c.setFont("Helvetica-Bold", 9)
        c.drawString(14.2*cm, height - 2.5*cm, "URLAUB (Tage)")
        c.setFont("Helvetica", 9)
        c.drawString(14.2*cm, height - 3.0*cm, f"Gesamtanspruch: {self.vac_stats[0]:g}")
        c.drawString(14.2*cm, height - 3.4*cm, f"Bereits genommen: {self.vac_stats[1]:g}")
        c.setFont("Helvetica-Bold", 9)
        c.drawString(14.2*cm, height - 3.8*cm, f"Resturlaub: {self.vac_stats[2]:g}")

        y = height - 5.5*cm
        x_offsets = [1*cm, 3.5*cm, 6*cm, 8*cm, 10*cm, 12*cm, 14*cm, 16*cm, 18*cm]
        headers = ["Datum", "Tag", "Start", "Ende", "Pause", "Art", "Ist", "Soll", "Diff"]
        
        c.setFont("Helvetica-Bold", 9)
        for i, h in enumerate(headers): c.drawString(x_offsets[i], y, h)
        c.line(1*cm, y - 0.2*cm, 20*cm, y - 0.2*cm)
        y -= 0.6*cm
        
        num_days = calendar.monthrange(self.year, self.month)[1]
        month_target, month_actual, month_diff = 0,0,0
        c.setFont("Helvetica", 9)
        
        for day in range(1, num_days + 1):
            d_obj = date(self.year, self.month, day)
            db_entry = db.get_entry(self.emp_id, d_obj.isoformat())
            
            start, end, pause, code = "", "", "", ""
            if db_entry: start, end, pause, code = db_entry[3], db_entry[4], db_entry[5], db_entry[6]
            
            actual = compute_work_hours(start, end, pause)
            target = self.targets[d_obj.weekday()]
            de_holidays = holidays.DE(subdiv=self.state, years=self.year)
            is_holiday = d_obj in de_holidays; is_weekend = d_obj.weekday() >= 5
            
            calc_actual = actual
            if is_holiday and target > 0: 
                target = 0.0; 
                if not code: code = "Feiertag"
            if code in ["Urlaub", "Krank"] and target > 0: calc_actual = target
            
            diff = calc_actual - target
            
            if is_holiday:
                c.setFillColor(colors.Color(1, 0.8, 0.8)); c.rect(1*cm, y - 0.1*cm, 19*cm, 0.5*cm, fill=1, stroke=0); c.setFillColor(colors.black)
            elif is_weekend:
                c.setFillColor(colors.Color(0.9, 0.9, 0.9)); c.rect(1*cm, y - 0.1*cm, 19*cm, 0.5*cm, fill=1, stroke=0); c.setFillColor(colors.black)

            c.drawString(x_offsets[0], y, f"{day:02d}.{self.month:02d}.")
            c.drawString(x_offsets[1], y, d_obj.strftime("%a"))
            if is_holiday:
                c.setFont("Helvetica-Oblique", 8); c.drawString(x_offsets[2], y, (de_holidays.get(d_obj) or "")[:20]); c.setFont("Helvetica", 9)
            else:
                c.drawString(x_offsets[2], y, start or "")
                c.drawString(x_offsets[3], y, end or "")
                c.drawString(x_offsets[4], y, pause or "")
                c.drawString(x_offsets[5], y, code or "")
                if actual > 0: c.drawString(x_offsets[6], y, f"{actual:.2f}")
                if target > 0: c.drawString(x_offsets[7], y, f"{target:.2f}")
                if abs(diff) > 0.01:
                    c.setFillColor(colors.red if diff < 0 else colors.green); c.drawString(x_offsets[8], y, f"{diff:+.2f}"); c.setFillColor(colors.black)

            month_target += target; month_actual += calc_actual; month_diff += diff
            y -= 0.5*cm
            if y < 2*cm: c.showPage(); y = height-2*cm; c.setFont("Helvetica", 9)

        y -= 0.5*cm
        c.line(1*cm, y + 0.3*cm, 20*cm, y + 0.3*cm)
        c.setFont("Helvetica-Bold", 9)
        c.drawString(x_offsets[0], y, "SUMME MONAT")
        c.drawString(x_offsets[6], y, f"{month_actual:.2f}")
        c.drawString(x_offsets[7], y, f"{month_target:.2f}")
        c.setFillColor(colors.red if month_diff < -0.01 else colors.green)
        c.drawString(x_offsets[8], y, f"{month_diff:+.2f}")
        c.setFillColor(colors.black)
        y -= 0.6*cm
        c.drawString(x_offsets[0], y, "SALDO VORMONAT:")
        c.drawString(x_offsets[2], y, f"{self.prev_balance:+.2f} h")
        c.drawString(x_offsets[4], y, "NEUER GESAMTSALDO:")
        final = self.prev_balance + month_diff
        c.setFont("Helvetica-Bold", 11)
        c.setFillColor(colors.red if final < 0 else colors.green)
        c.drawString(x_offsets[7], y, f"{final:+.2f} h")
        c.save()
        return self.filename

class YearlyPDF:
    def __init__(self, emp_data, year, vac_stats):
        self.emp_data = emp_data
        self.year = year
        self.vac_stats = vac_stats
        
        base_dir = get_documents_dir()
        self.filename = os.path.join(base_dir, f"Jahresbericht_{emp_data[2]}_{year}.pdf")
        
    def generate(self):
        c = canvas.Canvas(self.filename, pagesize=landscape(A4))
        width, height = landscape(A4)
        
        c.setFont("Helvetica-Bold", 18)
        c.drawString(2*cm, height - 2*cm, f"Jahresübersicht {self.year}")
        c.setFont("Helvetica", 12)
        c.drawString(2*cm, height - 3*cm, f"Mitarbeiter: {self.emp_data[1]} {self.emp_data[2]}")
        
        box_x = 23*cm
        c.setStrokeColor(colors.grey)
        c.rect(box_x, height - 4*cm, 5*cm, 1.8*cm)
        c.setFont("Helvetica-Bold", 9)
        c.drawString(box_x + 0.2*cm, height - 2.5*cm, "URLAUB (Stand Jahresende)")
        c.setFont("Helvetica", 9)
        c.drawString(box_x + 0.2*cm, height - 3.0*cm, f"Gesamtanspruch: {self.vac_stats[0]:g}")
        c.drawString(box_x + 0.2*cm, height - 3.4*cm, f"Bereits genommen: {self.vac_stats[1]:g}")
        c.setFont("Helvetica-Bold", 9)
        c.drawString(box_x + 0.2*cm, height - 3.8*cm, f"Resturlaub: {self.vac_stats[2]:g}")

        y = height - 5*cm
        x_start = 2*cm
        col_width = 3.5*cm
        headers = ["Monat", "Soll (h)", "Ist (h)", "Diff (h)", "Urlaub (Tage)", "Krank (Tage)"]
        
        c.setFont("Helvetica-Bold", 10)
        for i, h in enumerate(headers):
            c.drawString(x_start + i*col_width, y, h)
        
        c.line(x_start, y-0.2*cm, width-2*cm, y-0.2*cm)
        y -= 0.8*cm
        c.setFont("Helvetica", 10)
        
        totals = [0.0]*5
        
        for m in range(1, 13):
            month_name = calendar.month_name[m]
            num_days = calendar.monthrange(self.year, m)[1]
            m_soll, m_ist, m_diff, m_ur, m_kr = 0,0,0,0,0
            
            for d in range(1, num_days+1):
                d_obj = date(self.year, m, d)
                entry = db.get_entry(self.emp_data[0], d_obj.isoformat())
                
                targets = self.emp_data[9:16]
                target = targets[d_obj.weekday()]
                de_holidays = holidays.DE(subdiv=self.emp_data[6], years=self.year)
                
                if d_obj in de_holidays and target > 0: target = 0.0
                
                start, end, pause, code = 0,0,0, ""
                if entry:
                    def tf(t):
                        try: p=t.split(':'); return int(p[0]) + (int(p[1]) if len(p)>1 else 0)/60.0
                        except: return 0.0
                    start, end, pause = tf(entry[3]), tf(entry[4]), tf(entry[5])
                    code = entry[6]
                
                actual = (end-start-pause) if end>start else 0.0
                calc_actual = actual
                if code in ["Urlaub", "Krank"] and target > 0: calc_actual = target
                
                diff = calc_actual - target
                
                if code == "Urlaub": m_ur += 1
                if code == "Krank": m_kr += 1
                
                m_soll += target
                m_ist += calc_actual
                m_diff += diff
                
            c.drawString(x_start, y, month_name)
            c.drawString(x_start + col_width, y, f"{m_soll:.2f}")
            c.drawString(x_start + 2*col_width, y, f"{m_ist:.2f}")
            c.setFillColor(colors.red if m_diff < -0.01 else colors.green)
            c.drawString(x_start + 3*col_width, y, f"{m_diff:+.2f}")
            c.setFillColor(colors.black)
            c.drawString(x_start + 4*col_width, y, f"{m_ur:g}")
            c.drawString(x_start + 5*col_width, y, f"{m_kr:g}")
            
            totals[0]+=m_soll; totals[1]+=m_ist; totals[2]+=m_diff; totals[3]+=m_ur; totals[4]+=m_kr
            y -= 0.8*cm
            
        c.line(x_start, y+0.4*cm, width-2*cm, y+0.4*cm)
        c.setFont("Helvetica-Bold", 10)
        c.drawString(x_start, y, "GESAMT JAHR")
        c.drawString(x_start + col_width, y, f"{totals[0]:.2f}")
        c.drawString(x_start + 2*col_width, y, f"{totals[1]:.2f}")
        c.setFillColor(colors.red if totals[2] < -0.01 else colors.green)
        c.drawString(x_start + 3*col_width, y, f"{totals[2]:+.2f}")
        c.setFillColor(colors.black)
        c.drawString(x_start + 4*col_width, y, f"{totals[3]:g}")
        c.drawString(x_start + 5*col_width, y, f"{totals[4]:g}")
        
        c.save()
        return self.filename

# --- UI ---

class BulkEntryDialog(Adw.Window):
    def __init__(self, parent, emp_id, year, month, on_save):
        super().__init__(modal=True, transient_for=parent)
        self.set_title("Zeitraum eintragen (Urlaub/Krank)")
        self.set_default_size(650, 350)
        self.emp_id = emp_id; self.on_save = on_save
        
        page = Adw.PreferencesPage()
        group = Adw.PreferencesGroup()
        page.add(group)
        
        # Datum Start
        self.spin_day_start = Gtk.SpinButton.new_with_range(1, 31, 1); self.spin_day_start.set_value(1)
        self.spin_month_start = Gtk.SpinButton.new_with_range(1, 12, 1); self.spin_month_start.set_value(month)
        self.spin_year_start = Gtk.SpinButton.new_with_range(2020, 2030, 1); self.spin_year_start.set_value(year)
        row_start = Adw.ActionRow(title="Startdatum")
        box_s = Gtk.Box(spacing=5); box_s.append(self.spin_day_start); box_s.append(self.spin_month_start); box_s.append(self.spin_year_start)
        row_start.add_suffix(box_s)
        group.add(row_start)

        # Datum Ende
        self.spin_day_end = Gtk.SpinButton.new_with_range(1, 31, 1); self.spin_day_end.set_value(calendar.monthrange(year, month)[1])
        self.spin_month_end = Gtk.SpinButton.new_with_range(1, 12, 1); self.spin_month_end.set_value(month)
        self.spin_year_end = Gtk.SpinButton.new_with_range(2020, 2030, 1); self.spin_year_end.set_value(year)
        row_end = Adw.ActionRow(title="Enddatum")
        box_e = Gtk.Box(spacing=5); box_e.append(self.spin_day_end); box_e.append(self.spin_month_end); box_e.append(self.spin_year_end)
        row_end.add_suffix(box_e)
        group.add(row_end)

        # Code
        self.combo_code = Gtk.DropDown.new_from_strings(["Urlaub", "Krank", "Gleitzeit"])
        self.combo_code.set_selected(0)
        row_code = Adw.ActionRow(title="Typ")
        row_code.add_suffix(self.combo_code)
        group.add(row_code)

        header = Adw.HeaderBar()
        btn_do = Gtk.Button(label="Eintragen"); btn_do.add_css_class("suggested-action")
        btn_do.connect("clicked", self.process)
        header.set_title_widget(btn_do)
        
        self.set_content(Gtk.Box(orientation=Gtk.Orientation.VERTICAL))
        self.get_content().append(header)
        self.get_content().append(page)

    def process(self, widget):
        try:
            d_start = date(int(self.spin_year_start.get_value()), int(self.spin_month_start.get_value()), int(self.spin_day_start.get_value()))
            d_end = date(int(self.spin_year_end.get_value()), int(self.spin_month_end.get_value()), int(self.spin_day_end.get_value()))
        except ValueError: return

        if d_end < d_start: return

        code = ["Urlaub", "Krank", "Gleitzeit"][self.combo_code.get_selected()]
        curr = d_start
        while curr <= d_end:
            db.upsert_entry(self.emp_id, curr.isoformat(), "", "", "", code)
            curr += timedelta(days=1)
        
        self.on_save(); self.close()

class EmployeeDialog(Adw.Window):
    def __init__(self, parent, on_save_callback, employee_data=None):
        super().__init__(modal=True, transient_for=parent)
        self.set_title("Mitarbeiter bearbeiten" if employee_data else "Mitarbeiter anlegen")
        self.set_default_size(500, 750)
        self.on_save = on_save_callback
        self.employee_id = employee_data[0] if employee_data else None

        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        page = Adw.PreferencesPage()
        group_personal = Adw.PreferencesGroup(title="Persönliche Daten")
        page.add(group_personal)
        
        self.entries = {}
        fields = {"first_name": "Vorname", "last_name": "Nachname", "address": "Adresse", "phone": "Telefon", "email": "E-Mail"}
        current_values = {
            "first_name": employee_data[1] if employee_data else "",
            "last_name": employee_data[2] if employee_data else "",
            "address": employee_data[3] if employee_data else "",
            "phone": employee_data[4] if employee_data else "",
            "email": employee_data[5] if employee_data else ""
        }
        for key, label in fields.items():
            row = Adw.EntryRow(title=label); row.set_text(current_values[key])
            self.entries[key] = row; group_personal.add(row)

        self.combo_row = Adw.ComboRow(title="Bundesland")
        self.states = ["BW", "BY", "BE", "BB", "HB", "HH", "HE", "MV", "NI", "NW", "RP", "SL", "SN", "ST", "SH", "TH"]
        model = Gtk.StringList()
        for s in self.states: model.append(s)
        self.combo_row.set_model(model)
        if employee_data:
            try: self.combo_row.set_selected(self.states.index(employee_data[6]))
            except: self.combo_row.set_selected(1)
        else: self.combo_row.set_selected(1)
        group_personal.add(self.combo_row)

        group_vac = Adw.PreferencesGroup(title="Urlaub")
        page.add(group_vac)
        self.spin_vac = Adw.SpinRow.new_with_range(0, 100, 0.5); self.spin_vac.set_title("Jahresanspruch"); self.spin_vac.set_value(employee_data[7] if employee_data else 30.0)
        group_vac.add(self.spin_vac)
        self.spin_vac_over = Adw.SpinRow.new_with_range(0, 100, 0.5); self.spin_vac_over.set_title("Übertrag Vorjahr"); self.spin_vac_over.set_value(employee_data[8] if employee_data else 0.0)
        group_vac.add(self.spin_vac_over)

        group_hours = Adw.PreferencesGroup(title="Soll-Stunden")
        page.add(group_hours)
        self.spinners = []
        days = ["Montag", "Dienstag", "Mittwoch", "Donnerstag", "Freitag", "Samstag", "Sonntag"]
        for i, day_name in enumerate(days):
            row = Adw.ActionRow(title=day_name)
            spin = Gtk.SpinButton.new_with_range(0, 24, 0.5); spin.set_valign(Gtk.Align.CENTER)
            val = employee_data[9 + i] if employee_data else (8.0 if i < 5 else 0.0)
            spin.set_value(val); self.spinners.append(spin); row.add_suffix(spin); group_hours.add(row)

        header = Adw.HeaderBar()
        self.set_content(Gtk.Box(orientation=Gtk.Orientation.VERTICAL))
        self.get_content().append(header)
        self.get_content().append(page)
        
        btn_save = Gtk.Button(label="Speichern"); btn_save.add_css_class("suggested-action"); btn_save.connect("clicked", self.save)
        header.set_title_widget(btn_save)

        if self.employee_id:
            grp_danger = Adw.PreferencesGroup(title="Gefahrenzone"); page.add(grp_danger)
            btn_del = Gtk.Button(label="Mitarbeiter löschen"); btn_del.add_css_class("destructive-action")
            btn_del.connect("clicked", self.confirm_delete)
            row_del = Adw.ActionRow(); row_del.add_suffix(btn_del); grp_danger.add(row_del)

    def save(self, widget):
        data = {k: v.get_text() for k, v in self.entries.items()}
        state = self.states[self.combo_row.get_selected()]
        targets = [s.get_value() for s in self.spinners]
        vac_days = self.spin_vac.get_value(); vac_over = self.spin_vac_over.get_value()
        if self.employee_id:
            db.update_employee(self.employee_id, data['first_name'], data['last_name'], data['address'], data['phone'], data['email'], state, vac_days, vac_over, targets)
        else:
            db.add_employee(data['first_name'], data['last_name'], data['address'], data['phone'], data['email'], state, vac_days, vac_over, targets)
        self.on_save(); self.close()

    def confirm_delete(self, widget):
        dialog = Adw.MessageDialog(transient_for=self, heading="Mitarbeiter löschen?", body="Alle Daten werden gelöscht.")
        dialog.add_response("cancel", "Abbrechen"); dialog.add_response("delete", "Löschen"); dialog.set_response_appearance("delete", Adw.ResponseAppearance.DESTRUCTIVE)
        dialog.connect("response", self.on_delete_response); dialog.present()

    def on_delete_response(self, dialog, response):
        if response == "delete": db.delete_employee(self.employee_id); self.on_save(); self.close()

class DayRow(Gtk.ListBoxRow):
    def __init__(self, date_obj, emp_id, state_code, target_hours_map, parent_win=None):
        super().__init__()
        self.date_obj = date_obj; self.emp_id = emp_id; self.target_hours_map = target_hours_map; self.parent_win = parent_win
        self.db_entry = db.get_entry(emp_id, date_obj.isoformat())
        self.set_activatable(False); self.set_selectable(False)
        
        main_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=0)
        main_box.set_margin_top(8); main_box.set_margin_bottom(8); main_box.set_margin_start(12); main_box.set_margin_end(12)
        self.set_child(main_box)

        vbox_date = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=2)
        vbox_date.set_size_request(180, -1); vbox_date.set_valign(Gtk.Align.CENTER)
        day_name = date_obj.strftime("%A")
        lbl_title = Gtk.Label(label=f"{date_obj.day:02d}. {day_name}", xalign=0); lbl_title.add_css_class("body")
        lbl_subtitle = Gtk.Label(xalign=0); lbl_subtitle.add_css_class("caption"); lbl_subtitle.add_css_class("dim-label")
        
        de_holidays = holidays.DE(subdiv=state_code, years=date_obj.year)
        is_holiday = date_obj in de_holidays; is_weekend = date_obj.weekday() >= 5
        if is_holiday: lbl_subtitle.set_label(de_holidays.get(date_obj)); lbl_title.add_css_class("error")
        elif is_weekend: lbl_subtitle.set_label("Wochenende")
        vbox_date.append(lbl_title)
        if is_holiday or is_weekend: vbox_date.append(lbl_subtitle)
        main_box.append(vbox_date)

        box_inputs = Gtk.Box(spacing=15); box_inputs.set_valign(Gtk.Align.CENTER)
        self.ent_start = Gtk.Entry(placeholder_text="00:00", width_chars=8, xalign=0.5)
        self.ent_end = Gtk.Entry(placeholder_text="00:00", width_chars=8, xalign=0.5)
        self.ent_pause = Gtk.Entry(placeholder_text="00:00", width_chars=6, xalign=0.5)
        self.combo_code = Gtk.DropDown.new_from_strings(["Arbeit", "Urlaub", "Krank", "Gleitzeit"])
        self.combo_code.set_valign(Gtk.Align.CENTER)
        
        if self.db_entry:
            self.ent_start.set_text(self.db_entry[3] or "")
            self.ent_end.set_text(self.db_entry[4] or "")
            self.ent_pause.set_text(self.db_entry[5] or "")
            if self.db_entry[6] in ["Arbeit", "Urlaub", "Krank", "Gleitzeit"]:
                self.combo_code.set_selected(["Arbeit", "Urlaub", "Krank", "Gleitzeit"].index(self.db_entry[6]))

        self.lbl_delta = Gtk.Label(label="0.00", width_chars=6, xalign=1)
        self.ent_start.connect("changed", self.on_change); self.ent_end.connect("changed", self.on_change)
        self.ent_pause.connect("changed", self.on_change); self.combo_code.connect("notify::selected-item", self.on_change)
        box_inputs.append(self.ent_start); box_inputs.append(self.ent_end); box_inputs.append(self.ent_pause)
        box_inputs.append(self.combo_code); box_inputs.append(self.lbl_delta)
        main_box.append(box_inputs)
        self.calculate_time()

    def calculate_time(self):
        # Neue Logik: Nachtschicht
        actual = compute_work_hours(self.ent_start.get_text(), self.ent_end.get_text(), self.ent_pause.get_text())
        code = ["Arbeit", "Urlaub", "Krank", "Gleitzeit"][self.combo_code.get_selected()]
        target = self.target_hours_map[self.date_obj.weekday()]
        
        if self.date_obj in holidays.DE(subdiv='BY', years=self.date_obj.year) and target > 0: target = 0.0
        
        if (code in ["Urlaub", "Krank"]) and target > 0: delta = 0.0
        else: delta = actual - target
        
        if (actual > 0 or delta != 0) and not (code in ["Urlaub", "Krank"] and delta == 0):
            self.lbl_delta.set_label(f"{delta:+.2f}")
            self.lbl_delta.add_css_class("error" if delta < 0 else "success")
            self.lbl_delta.remove_css_class("success" if delta < 0 else "error")
        else: 
            self.lbl_delta.set_label(""); self.lbl_delta.remove_css_class("error"); self.lbl_delta.remove_css_class("success")

    def on_change(self, widget, *args):
        self.calculate_time()
        code = ["Arbeit", "Urlaub", "Krank", "Gleitzeit"][self.combo_code.get_selected()]
        db.upsert_entry(self.emp_id, self.date_obj.isoformat(), self.ent_start.get_text(), self.ent_end.get_text(), self.ent_pause.get_text(), code)
        if self.parent_win: self.parent_win.update_vacation_stats()

# --- STATS VIEW ---
class StatsView(Gtk.DrawingArea):
    def __init__(self):
        super().__init__()
        self.set_draw_func(self.draw, None)
        self.data = []; self.vac_data = (0,0)

    def set_data(self, balances, vac_stats):
        self.data = balances; self.vac_data = (vac_stats[0], vac_stats[1]); self.queue_draw()

    def draw(self, area, cr, width, height, user_data):
        cr.set_source_rgb(0.95, 0.95, 0.95); cr.paint()
        cr.set_source_rgb(0.2, 0.2, 0.2); cr.set_font_size(14); cr.move_to(20, 30); cr.show_text("Urlaubsstatus:")
        
        bar_w = width - 40; bar_h = 30; bar_y = 45
        total, taken = self.vac_data
        pct = min(1.0, taken / total) if total > 0 else 0
        
        cr.set_line_width(1); cr.set_source_rgb(0.5,0.5,0.5); cr.rectangle(20, bar_y, bar_w, bar_h); cr.stroke()
        cr.set_source_rgb(0.2, 0.6, 0.8); cr.rectangle(20, bar_y, bar_w * pct, bar_h); cr.fill()
        cr.set_source_rgb(0,0,0); cr.set_font_size(12); cr.move_to(30, bar_y + 20); cr.show_text(f"{taken:g} von {total:g} Tagen genommen")

        graph_y = 120; graph_h = height - graph_y - 40
        if graph_h < 50: return
        cr.set_source_rgb(0.2, 0.2, 0.2); cr.set_font_size(14); cr.move_to(20, 110); cr.show_text("Überstunden-Entwicklung (Monatssaldo):")
        
        zero_y = graph_y + graph_h / 2
        cr.set_source_rgb(0.7, 0.7, 0.7); cr.set_line_width(1); cr.move_to(20, zero_y); cr.line_to(width-20, zero_y); cr.stroke()
        col_w = (width - 40) / 13
        max_val = max(max([abs(x) for x in self.data]) if self.data else 1, 10)
        scale = (graph_h / 2 - 10) / max_val
        months = ["Jan", "Feb", "Mär", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez"]
        
        for i, val in enumerate(self.data):
            x = 20 + i * col_w + 10; h = val * scale
            if val >= 0:
                cr.set_source_rgb(0.3, 0.7, 0.3); cr.rectangle(x, zero_y - h, col_w - 10, h); cr.fill()
                cr.set_source_rgb(0,0,0); cr.set_font_size(10); cr.move_to(x, zero_y - h - 5); cr.show_text(f"{val:.1f}")
            else:
                cr.set_source_rgb(0.8, 0.3, 0.3); cr.rectangle(x, zero_y, col_w - 10, abs(h)); cr.fill()
                cr.set_source_rgb(0,0,0); cr.set_font_size(10); cr.move_to(x, zero_y + abs(h) + 10); cr.show_text(f"{val:.1f}")
            cr.set_source_rgb(0.4,0.4,0.4); cr.move_to(x, height - 10); cr.show_text(months[i])

class MainWindow(Adw.ApplicationWindow):
    def __init__(self, app):
        super().__init__(application=app, title="Arbeitszeiterfassung")
        self.set_default_size(1100, 850)
        main_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        self.toast_overlay = Adw.ToastOverlay(); self.toast_overlay.set_child(main_box)
        self.set_content(self.toast_overlay)
        
        header = Adw.HeaderBar()
        box_header = Gtk.Box(spacing=10)
        try:
            script_dir = os.path.dirname(os.path.abspath(__file__))
            icon_path = os.path.join(script_dir, "icons", "logo.png")
            if os.path.exists(icon_path):
                texture = Gdk.Texture.new_from_filename(icon_path)
                img_logo = Gtk.Image.new_from_paintable(texture); img_logo.set_pixel_size(24); box_header.append(img_logo)
        except: pass

        lbl_app_title = Gtk.Label(label="Arbeitszeiterfassung", xalign=0); lbl_app_title.add_css_class("title-4"); box_header.append(lbl_app_title)
        btn_info = Gtk.Button(icon_name="dialog-information-symbolic"); btn_info.add_css_class("flat"); btn_info.connect("clicked", self.show_about)
        box_header.append(btn_info)
        header.pack_start(box_header)
        main_box.append(header)
        
        self.view_stack = Adw.ViewStack(); self.view_stack.set_vexpand(True)
        main_box.append(self.view_stack)
        self.view_stack.connect("notify::visible-child", self.on_tab_changed)
        header.set_title_widget(Adw.ViewSwitcherTitle(stack=self.view_stack, title="Zeiterfassung"))

        # TAB MITARBEITER
        self.page_employees = Adw.StatusPage(title="Mitarbeiter", icon_name="avatar-default-symbolic")
        emp_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        toolbar = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=15)
        toolbar.set_margin_top(10); toolbar.set_margin_bottom(10); toolbar.set_halign(Gtk.Align.CENTER)
        btn_add = Gtk.Button(label="Neuer Mitarbeiter", icon_name="list-add-symbolic"); btn_add.add_css_class("pill")
        btn_add.connect("clicked", self.open_add_employee)
        toolbar.append(btn_add)
        toolbar.append(Gtk.Separator(orientation=Gtk.Orientation.VERTICAL))
        btn_backup = Gtk.Button(label="Backup", icon_name="document-save-symbolic")
        btn_backup.connect("clicked", self.do_backup)
        toolbar.append(btn_backup)
        emp_box.append(toolbar)
        self.emp_listbox = Gtk.ListBox(); self.emp_listbox.set_selection_mode(Gtk.SelectionMode.NONE); self.emp_listbox.add_css_class("boxed-list")
        frame_scroll = Gtk.ScrolledWindow(); clamp = Adw.Clamp(maximum_size=700); clamp.set_child(self.emp_listbox)
        frame_scroll.set_child(clamp); frame_scroll.set_vexpand(True)
        emp_box.append(frame_scroll)
        self.view_stack.add_titled_with_icon(emp_box, "employees", "Mitarbeiter", "avatar-default-symbolic")

        # TAB ERFASSUNG
        self.page_times = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        ctrl_box = Gtk.Box(spacing=10); ctrl_box.set_margin_top(20); ctrl_box.set_margin_bottom(25); ctrl_box.set_halign(Gtk.Align.CENTER)
        self.lbl_current_emp = Gtk.Label(label="Kein Mitarbeiter gewählt"); self.lbl_current_emp.add_css_class("title-3")
        self.spin_year = Gtk.SpinButton.new_with_range(2020, 2030, 1); self.spin_year.set_value(datetime.now().year)
        self.spin_year.connect("value-changed", self.refresh_calendar)
        self.combo_month = Gtk.DropDown.new_from_strings(list(calendar.month_name)[1:]); self.combo_month.set_selected(datetime.now().month - 1)
        self.combo_month.connect("notify::selected-item", self.refresh_calendar)
        
        # Buttons mit Tooltips und Symbolic Icons
        btn_bulk = Gtk.Button(icon_name="x-office-calendar-symbolic")
        btn_bulk.set_tooltip_text("Zeitraum eintragen (Urlaub/Krank)")
        btn_bulk.connect("clicked", self.open_bulk_dialog)
        
        btn_pdf = Gtk.Button(label="Monat", icon_name="printer-symbolic")
        btn_pdf.set_tooltip_text("Monatsbericht als PDF speichern")
        btn_pdf.connect("clicked", self.create_pdf)
        
        btn_pdf_year = Gtk.Button(label="Jahr", icon_name="x-office-calendar-symbolic")
        btn_pdf_year.set_tooltip_text("Jahresübersicht als PDF speichern")
        btn_pdf_year.connect("clicked", self.create_yearly_pdf)
        
        ctrl_box.append(self.lbl_current_emp); ctrl_box.append(Gtk.Separator(orientation=Gtk.Orientation.VERTICAL))
        ctrl_box.append(self.combo_month); ctrl_box.append(self.spin_year)
        ctrl_box.append(Gtk.Separator(orientation=Gtk.Orientation.VERTICAL))
        ctrl_box.append(btn_bulk) # Bulk Button
        ctrl_box.append(Gtk.Separator(orientation=Gtk.Orientation.VERTICAL))
        ctrl_box.append(btn_pdf); ctrl_box.append(btn_pdf_year)
        self.page_times.append(ctrl_box)

        self.vac_info_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=20)
        self.vac_info_box.set_halign(Gtk.Align.CENTER); self.vac_info_box.set_margin_bottom(30)
        self.vac_info_box.add_css_class("card"); self.vac_info_box.set_margin_start(50); self.vac_info_box.set_margin_end(50)
        self.lbl_vac_avail = Gtk.Label(label="Anspruch: -"); self.lbl_vac_avail.set_margin_start(10); self.lbl_vac_avail.set_margin_top(5); self.lbl_vac_avail.set_margin_bottom(5)
        self.lbl_vac_taken = Gtk.Label(label="Genommen: -"); self.lbl_vac_rest = Gtk.Label(label="Rest: -"); self.lbl_vac_rest.set_margin_end(10); self.lbl_vac_rest.add_css_class("accent")
        self.vac_info_box.append(self.lbl_vac_avail); self.vac_info_box.append(Gtk.Separator(orientation=Gtk.Orientation.VERTICAL))
        self.vac_info_box.append(self.lbl_vac_taken); self.vac_info_box.append(Gtk.Separator(orientation=Gtk.Orientation.VERTICAL)); self.vac_info_box.append(self.lbl_vac_rest)
        self.page_times.append(self.vac_info_box)
        
        header_clamp = Adw.Clamp(maximum_size=1000); header_box = Gtk.Box(spacing=0)
        header_box.set_margin_start(12); header_box.set_margin_end(12); header_box.set_margin_bottom(5)
        lbl_date = Gtk.Label(label="Datum", xalign=0); lbl_date.set_size_request(180, -1); lbl_date.add_css_class("heading")
        header_box.append(lbl_date)
        box_right = Gtk.Box(spacing=15)
        for t, w in [("Start", 8), ("Ende", 8), ("Pause", 6), ("Art", 8), ("Diff", 6)]: l = Gtk.Label(label=t, width_chars=w); l.add_css_class("heading"); box_right.append(l)
        header_box.append(box_right); header_clamp.set_child(header_box); self.page_times.append(header_clamp)
        scrolled = Gtk.ScrolledWindow(); self.days_listbox = Gtk.ListBox(); self.days_listbox.set_selection_mode(Gtk.SelectionMode.NONE); self.days_listbox.add_css_class("boxed-list")
        clamp_list = Adw.Clamp(maximum_size=1000); clamp_list.set_child(self.days_listbox)
        scrolled.set_child(clamp_list); scrolled.set_vexpand(True); self.page_times.append(scrolled)
        self.view_stack.add_titled_with_icon(self.page_times, "times", "Erfassung", "accessories-text-editor-symbolic")
        
        # TAB STATISTIK
        self.page_stats = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        self.stats_view = StatsView()
        self.stats_view.set_vexpand(True)
        self.page_stats.append(self.stats_view)
        self.view_stack.add_titled_with_icon(self.page_stats, "stats", "Statistik", "view-list-symbolic")

        self.selected_emp_id = None; self.refresh_employee_list()

    def show_about(self, widget):
        dialog = Adw.AboutWindow(transient_for=self, application_name="Arbeitszeiterfassung", application_icon="x-office-calendar", developer_name="armin@Pinguin-TV", version="1.2", copyright="© 2025 armin@Pinguin-TV", comments="Verwendung auf eigene Gefahr, es wird jede Haftung ausgeschlossen.")
        dialog.present()

    def on_tab_changed(self, stack, param):
        name = stack.get_visible_child_name()
        if name == "employees": self.refresh_employee_list()
        if name == "stats" and self.selected_emp_id:
            year = int(self.spin_year.get_value())
            balances = db.get_monthly_balances_for_year(self.selected_emp_id, year)
            emp = db.get_employee_by_id(self.selected_emp_id)
            taken = db.get_taken_vacation_days(self.selected_emp_id, year)
            total = emp[7] + emp[8]; rest = total - taken
            self.stats_view.set_data(balances, (total, taken, rest))

    def open_add_employee(self, widget): EmployeeDialog(self, self.refresh_employee_list).present()
    def open_edit_employee(self, widget): EmployeeDialog(self, self.refresh_employee_list, db.get_employee_by_id(widget.emp_id)).present()
    
    # NEU: Backup mit Ordner-Auswahl
    def do_backup(self, widget):
        dialog = Gtk.FileChooserNative(title="Backup Speicherort wählen", parent=self, action=Gtk.FileChooserAction.SELECT_FOLDER, accept_label="Speichern", cancel_label="Abbrechen")
        dialog.connect("response", self.on_backup_response)
        dialog.show()

    def on_backup_response(self, dialog, response):
        if response == Gtk.ResponseType.ACCEPT:
            folder_file = dialog.get_file()
            folder_path = folder_file.get_path()
            try:
                dest = db.create_backup(folder_path)
                self.toast_overlay.add_toast(Adw.Toast.new(f"Backup erfolgreich: {os.path.basename(dest)}"))
            except Exception as e: self.toast_overlay.add_toast(Adw.Toast.new(f"Fehler: {e}"))
        dialog.destroy()

    def open_bulk_dialog(self, widget):
        if not self.selected_emp_id: return
        year = int(self.spin_year.get_value()); month = self.combo_month.get_selected() + 1
        BulkEntryDialog(self, self.selected_emp_id, year, month, self.refresh_calendar).present()

    def select_employee(self, widget):
        self.selected_emp_id = widget.emp_id
        data = db.get_employee_by_id(widget.emp_id)
        if not data: return
        self.selected_emp_state = data[6]; self.selected_emp_targets = data[9:16]
        self.selected_emp_vac_days = data[7]; self.selected_emp_vac_over = data[8]
        self.lbl_current_emp.set_label(f"{data[1]} {data[2]}")
        self.view_stack.set_visible_child_name("times")
        self.refresh_calendar(); self.update_vacation_stats()

    def update_vacation_stats(self):
        if not self.selected_emp_id: return
        year = int(self.spin_year.get_value())
        taken = db.get_taken_vacation_days(self.selected_emp_id, year)
        total_avail = self.selected_emp_vac_days + self.selected_emp_vac_over
        rest = total_avail - taken
        self.lbl_vac_avail.set_label(f"Anspruch: {self.selected_emp_vac_days:g} + {self.selected_emp_vac_over:g}")
        self.lbl_vac_taken.set_label(f"Genommen ({year}): {taken:g}")
        self.lbl_vac_rest.set_label(f"Verfügbar: {rest:g}")
        if rest < 0: self.lbl_vac_rest.add_css_class("error")
        else: self.lbl_vac_rest.remove_css_class("error")

    def refresh_employee_list(self):
        while child := self.emp_listbox.get_first_child(): self.emp_listbox.remove(child)
        for emp in db.get_employees():
            row = Adw.ActionRow(); row.set_title(f"{emp[1]} {emp[2]}")
            balance = db.calculate_balance(emp[0]); balance_str = f"{balance:+.2f} h"
            row.set_subtitle(f"Konto: {balance_str} | Urlaub: {emp[7]} Tg")
            if balance < 0: row.add_css_class("error")
            box = Gtk.Box(spacing=10); box.set_valign(Gtk.Align.CENTER)
            btn_edit = Gtk.Button(icon_name="document-edit-symbolic"); btn_edit.add_css_class("flat")
            btn_edit.emp_id = emp[0]; btn_edit.connect("clicked", self.open_edit_employee)
            box.append(btn_edit)
            btn_sel = Gtk.Button(icon_name="go-next-symbolic"); btn_sel.add_css_class("flat")
            btn_sel.emp_id = emp[0]; btn_sel.connect("clicked", self.select_employee)
            box.append(btn_sel)
            row.add_suffix(box); self.emp_listbox.append(row)

    def refresh_calendar(self, *args):
        if not self.selected_emp_id: return
        year = int(self.spin_year.get_value()); month = self.combo_month.get_selected() + 1
        while child := self.days_listbox.get_first_child(): self.days_listbox.remove(child)
        for day in range(1, calendar.monthrange(year, month)[1] + 1):
            self.days_listbox.append(DayRow(date(year, month, day), self.selected_emp_id, self.selected_emp_state, self.selected_emp_targets, self))
        self.update_vacation_stats()

    def create_pdf(self, widget):
        if not self.selected_emp_id: return
        year = int(self.spin_year.get_value()); month = self.combo_month.get_selected() + 1
        emp_data = db.get_employee_by_id(self.selected_emp_id)
        taken = db.get_taken_vacation_days(self.selected_emp_id, year)
        total = emp_data[7] + emp_data[8]; rest = total - taken
        vac_stats = (total, taken, rest)
        prev_balance = db.calculate_balance(self.selected_emp_id, until_date_str=date(year, month, 1).isoformat())
        pdf = MonthlyPDF(emp_data, year, month, vac_stats, prev_balance)
        filename = pdf.generate()
        self.toast_overlay.add_toast(Adw.Toast.new(f"PDF erstellt: {filename}"))
        try: subprocess.call(["xdg-open", filename])
        except: pass

    def create_yearly_pdf(self, widget):
        if not self.selected_emp_id: return
        year = int(self.spin_year.get_value())
        emp_data = db.get_employee_by_id(self.selected_emp_id)
        taken = db.get_taken_vacation_days(self.selected_emp_id, year)
        total = emp_data[7] + emp_data[8]; rest = total - taken
        vac_stats = (total, taken, rest)
        pdf = YearlyPDF(emp_data, year, vac_stats)
        filename = pdf.generate()
        self.toast_overlay.add_toast(Adw.Toast.new(f"Jahresbericht erstellt: {filename}"))
        try: subprocess.call(["xdg-open", filename])
        except: pass

class MyApp(Adw.Application):
    def __init__(self, **kwargs):
        super().__init__(**kwargs); self.connect('activate', self.on_activate)
    def on_activate(self, app): MainWindow(app).present()

if __name__ == "__main__":
    app = MyApp(application_id="com.example.arbeitszeit")
    app.run(sys.argv)