#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
db.py
SQLite-Datenbankmodul für FinanzPlaner.
Enthält Modelle, CRUD-Funktionen und Filter.
"""

import sqlite3
from dataclasses import dataclass
from typing import List, Optional, Dict
from datetime import datetime, date
import calendar


DB_FILE = "haushalt.db"


# ───────────────────────────────────────────────────────────────
#   Daten-Klassen
# ───────────────────────────────────────────────────────────────

@dataclass
class Transaction:
    id: Optional[int]
    date: str
    description: str
    category: str
    amount: float
    is_income: bool   # True = Einnahme


@dataclass
class Budget:
    id: Optional[int]
    category: str
    monthly_limit: float


@dataclass
class RecurringTransaction:
    id: Optional[int]
    description: str
    category: str
    amount: float
    is_income: bool
    interval_months: int
    next_date: str      # 'YYYY-MM-DD'


# ───────────────────────────────────────────────────────────────
#   Datenbank-Klasse
# ───────────────────────────────────────────────────────────────

class Database:
    def __init__(self, path: str = DB_FILE):
        self.path = path
        self.conn = sqlite3.connect(self.path)
        self.conn.row_factory = sqlite3.Row
        self._init_db()

    # ---------------------------
    # Erstelle Tabellen
    # ---------------------------
    def _init_db(self):
        cur = self.conn.cursor()

        cur.execute("""
            CREATE TABLE IF NOT EXISTS transactions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                date TEXT NOT NULL,
                description TEXT NOT NULL,
                category TEXT NOT NULL,
                amount REAL NOT NULL,
                is_income INTEGER NOT NULL
            )
        """)

        cur.execute("""
            CREATE TABLE IF NOT EXISTS budgets (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                category TEXT NOT NULL UNIQUE,
                monthly_limit REAL NOT NULL
            )
        """)

        # NEU: Wiederkehrende Buchungen
        cur.execute("""
            CREATE TABLE IF NOT EXISTS recurring_transactions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                description TEXT NOT NULL,
                category TEXT NOT NULL,
                amount REAL NOT NULL,
                is_income INTEGER NOT NULL,
                interval_months INTEGER NOT NULL,
                next_date TEXT NOT NULL
            )
        """)

        self.conn.commit()

    # ---------------------------
    # TRANSAKTIONEN
    # ---------------------------

    def add_transaction(self, tx: Transaction) -> int:
        cur = self.conn.cursor()
        cur.execute("""
            INSERT INTO transactions (date, description, category, amount, is_income)
            VALUES (?, ?, ?, ?, ?)
        """, (tx.date, tx.description, tx.category, tx.amount, int(tx.is_income)))
        self.conn.commit()
        return cur.lastrowid

    def update_transaction(self, tx: Transaction):
        cur = self.conn.cursor()
        cur.execute("""
            UPDATE transactions
            SET date = ?, description = ?, category = ?, amount = ?, is_income = ?
            WHERE id = ?
        """, (tx.date, tx.description, tx.category, tx.amount, int(tx.is_income), tx.id))
        self.conn.commit()

    def delete_transaction(self, tx_id: int):
        cur = self.conn.cursor()
        cur.execute("DELETE FROM transactions WHERE id = ?", (tx_id,))
        self.conn.commit()

    def get_all_transactions(self) -> List[Transaction]:
        cur = self.conn.cursor()
        rows = cur.execute("""
            SELECT id, date, description, category, amount, is_income
            FROM transactions ORDER BY id ASC
        """).fetchall()

        return [
            Transaction(
                id=row["id"],
                date=row["date"],
                description=row["description"],
                category=row["category"],
                amount=row["amount"],
                is_income=bool(row["is_income"])
            )
            for row in rows
        ]

    # ---------------------------
    # FILTER
    # ---------------------------

    def get_transactions_by_category(self, category: str) -> List[Transaction]:
        cur = self.conn.cursor()
        rows = cur.execute("""
            SELECT *
            FROM transactions
            WHERE category = ?
            ORDER BY id ASC
        """, (category,)).fetchall()

        return [
            Transaction(
                id=row["id"],
                date=row["date"],
                description=row["description"],
                category=row["category"],
                amount=row["amount"],
                is_income=bool(row["is_income"])
            )
            for row in rows
        ]

    def get_transactions_by_month(self, year: int, month: int) -> List[Transaction]:
        """
        Achtung: Diese Funktion geht davon aus, dass das Datumsformat 'YYYY-MM-...'
        ist. In der App verwenden wir für Filter meist parse_date(...) + Python-Logik.
        """
        prefix = f"{year:04d}-{month:02d}"
        cur = self.conn.cursor()
        rows = cur.execute("""
            SELECT *
            FROM transactions
            WHERE date LIKE ?
            ORDER BY id ASC
        """, (prefix + "%",)).fetchall()

        return [
            Transaction(
                id=row["id"],
                date=row["date"],
                description=row["description"],
                category=row["category"],
                amount=row["amount"],
                is_income=bool(row["is_income"])
            )
            for row in rows
        ]

    # ---------------------------
    # BUDGETS
    # ---------------------------

    def upsert_budget(self, category: str, monthly_limit: float):
        cur = self.conn.cursor()
        cur.execute("""
            INSERT INTO budgets (category, monthly_limit)
            VALUES (?, ?)
            ON CONFLICT(category) DO UPDATE SET
            monthly_limit = excluded.monthly_limit
        """, (category, monthly_limit))
        self.conn.commit()

    def get_budgets(self) -> Dict[str, Budget]:
        cur = self.conn.cursor()
        rows = cur.execute("SELECT * FROM budgets").fetchall()
        budgets = {}
        for row in rows:
            budgets[row["category"]] = Budget(
                id=row["id"],
                category=row["category"],
                monthly_limit=row["monthly_limit"]
            )
        return budgets

    # ---------------------------
    # Wiederkehrende Buchungen
    # ---------------------------

    def add_recurring(self, rt: RecurringTransaction) -> int:
        cur = self.conn.cursor()
        cur.execute("""
            INSERT INTO recurring_transactions
            (description, category, amount, is_income, interval_months, next_date)
            VALUES (?, ?, ?, ?, ?, ?)
        """, (rt.description, rt.category, rt.amount, int(rt.is_income),
              rt.interval_months, rt.next_date))
        self.conn.commit()
        return cur.lastrowid

    def update_recurring(self, rt: RecurringTransaction):
        cur = self.conn.cursor()
        cur.execute("""
            UPDATE recurring_transactions
            SET description = ?, category = ?, amount = ?, is_income = ?,
                interval_months = ?, next_date = ?
            WHERE id = ?
        """, (
            rt.description,
            rt.category,
            rt.amount,
            int(rt.is_income),
            rt.interval_months,
            rt.next_date,
            rt.id,
        ))
        self.conn.commit()

    def delete_recurring(self, rid: int):
        cur = self.conn.cursor()
        cur.execute("DELETE FROM recurring_transactions WHERE id = ?", (rid,))
        self.conn.commit()

    def get_recurring(self) -> List[RecurringTransaction]:
        cur = self.conn.cursor()
        rows = cur.execute("""
            SELECT *
            FROM recurring_transactions
            ORDER BY id ASC
        """).fetchall()

        return [
            RecurringTransaction(
                id=row["id"],
                description=row["description"],
                category=row["category"],
                amount=row["amount"],
                is_income=bool(row["is_income"]),
                interval_months=row["interval_months"],
                next_date=row["next_date"],
            )
            for row in rows
        ]

    def _add_months(self, d: date, months: int) -> date:
        """
        Hilfsfunktion: fügt einem Datum eine Anzahl Monate hinzu.
        """
        month = d.month - 1 + months
        year = d.year + month // 12
        month = month % 12 + 1
        last_day = calendar.monthrange(year, month)[1]
        day = min(d.day, last_day)
        return date(year, month, day)

    def apply_recurring(self, up_to: date):
        """
        Wendet alle fälligen wiederkehrenden Buchungen bis einschließlich 'up_to' an.
        Für jede fällige Periode wird eine Transaktion in 'transactions' angelegt und
        das Feld 'next_date' entsprechend weitergeschoben.
        """
        cur = self.conn.cursor()
        rows = cur.execute("""
            SELECT *
            FROM recurring_transactions
        """).fetchall()

        changed_any = False

        for row in rows:
            rid = row["id"]
            try:
                next_d = datetime.strptime(row["next_date"], "%Y-%m-%d").date()
            except Exception:
                # Ungültiges Datum -> überspringen
                continue

            interval = int(row["interval_months"])
            if interval <= 0:
                continue

            updated_date = next_d
            while updated_date <= up_to:
                tx = Transaction(
                    id=None,
                    date=updated_date.strftime("%Y-%m-%d"),
                    description=row["description"],
                    category=row["category"],
                    amount=row["amount"],
                    is_income=bool(row["is_income"]),
                )
                self.add_transaction(tx)
                updated_date = self._add_months(updated_date, interval)
                changed_any = True

            if updated_date != next_d:
                cur.execute(
                    "UPDATE recurring_transactions SET next_date = ? WHERE id = ?",
                    (updated_date.strftime("%Y-%m-%d"), rid)
                )

        if changed_any:
            self.conn.commit()

    # ---------------------------
    # Hilfsfunktionen
    # ---------------------------

    def get_month_sum(self) -> Dict[str, float]:
        """
        Gibt zurück: {"YYYY-MM": summe}
        """
        cur = self.conn.cursor()
        rows = cur.execute("""
            SELECT substr(date, 1, 7) AS month, SUM(amount) AS total
            FROM transactions
            WHERE is_income = 0
            GROUP BY month
        """).fetchall()
        return {row["month"]: row["total"] for row in rows}

    def calculate_category_sums(self) -> Dict[str, float]:
        cur = self.conn.cursor()
        rows = cur.execute("""
            SELECT category, SUM(amount) AS total
            FROM transactions
            WHERE is_income = 0
            GROUP BY category
        """).fetchall()
        return {row["category"]: row["total"] for row in rows}

    def close(self):
        self.conn.close()
