#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
FinanzPlaner — GTK4 + libadwaita Haushaltsbuch
"""

import gi
gi.require_version("Gtk", "4.0")
gi.require_version("Adw", "1")

from gi.repository import Gtk, Adw, Gio, Gdk

from db import Database, Transaction, RecurringTransaction
from utils import (
    parse_date,
    format_euro,
    current_month_key,
    month_key,
)
from charts import DonutChartWidget, BarChartWidget
from widgets import (
    TransactionListWidget,
    EditTransactionDialog,
    DeleteConfirmationDialog,
    FilterBannerWidget,
)

import csv
import shutil
from datetime import date, datetime
from collections import defaultdict

# PDF-Export (optional)
try:
    from reportlab.pdfgen import canvas as pdfcanvas
    from reportlab.lib.pagesizes import A4
    from reportlab.lib.units import cm
except Exception:
    pdfcanvas = None
    A4 = None
    cm = None

# Hilfsliste für Monatsnamen
MONTH_NAMES = [
    "Januar", "Februar", "März", "April", "Mai", "Juni",
    "Juli", "August", "September", "Oktober", "November", "Dezember"
]


# ╔══════════════════════════════════════════╗
#   MAIN WINDOW
# ╚══════════════════════════════════════════╝

class MainWindow(Adw.ApplicationWindow):
    def __init__(self, app):
        super().__init__(application=app)
        self.set_title("FinanzPlaner")
        self.set_default_size(1300, 1020)

        # Flag zur Vermeidung von Endlos-Schleifen beim Button-Umschalten
        self._is_navigating = False
        # Flag, um Signal-Loops bei den Filter-Dropdowns zu verhindern
        self._updating_filter_combo = False

        # DB + Daten
        self.db = Database()
        # Wiederkehrende Buchungen bis heute anwenden
        self.db.apply_recurring(date.today())
        self.transactions = self.db.get_all_transactions()
        self.budgets = self.db.get_budgets()

        # Kategorie-Store für Autovervollständigung
        self.category_store = Gtk.ListStore(str)

        # Filterstatus
        self.active_category_filter = None
        self.active_month_filter = None  # Format: (year, month) z.B. (2025, 11)

        # Styling
        self._init_css()

        # UI
        self._build_ui()

        # Daten laden
        self.refresh_all()

    # -----------------------------------------
    # CSS (libadwaita-pure)
    # -----------------------------------------
    def _init_css(self):
        css = """
        .summary-card-title {
            font-size: 14px;
            opacity: .7;
        }
        .summary-card-value {
            font-size: 22px;
            font-weight: 600;
        }
        .section-title {
            font-size: 18px;
            font-weight: 600;
            margin-top: 4px;
            margin-bottom: 4px;
        }
        .amount-income {
            color: @success_fg_color;
            font-weight: 700;
        }
        .amount-expense {
            color: @error_fg_color;
            font-weight: 700;
        }
        .list-header {
            font-weight: 700;
            opacity: .7;
            font-size: 12px;
        }

        /* Allgemeine Card-Optik */
        .card, card, adwbin {
            border-radius: 14px;
            background-color: @window_bg_color;
            box-shadow: 0 2px 6px alpha(black, 0.18);
        }

        /* Hover für ListBox Rows */
        row:hover {
            background-color: alpha(@accent_bg_color, 0.10);
            transition: 150ms ease;
        }

        /* Schnellkategorie-Buttons */
        .quick-cat {
            border-radius: 20px;
            padding: 6px 14px;
            background-color: alpha(@accent_bg_color, 0.15);
        }
        .quick-cat:hover {
            background-color: alpha(@accent_bg_color, 0.25);
        }

        /* Budget-Ampel */
        .budget-ok {
            color: @success_fg_color;
        }
        .budget-warn {
            color: @warning_fg_color;
        }
        .budget-over {
            color: @error_fg_color;
            font-weight: 700;
        }
        """

        provider = Gtk.CssProvider()
        provider.load_from_data(css)
        Gtk.StyleContext.add_provider_for_display(
            Gdk.Display.get_default(), provider, Gtk.STYLE_PROVIDER_PRIORITY_USER
        )

    # ╔══════════════════════════════╗
    #   UI Aufbau
    # ╚══════════════════════════════╝
    def _build_ui(self):
        self.toolbar_view = Adw.ToolbarView()
        self.set_content(self.toolbar_view)

        # Headerbar
        header = Adw.HeaderBar()
        self.toolbar_view.add_top_bar(header)

        title = Adw.WindowTitle.new("FinanzPlaner", "Haushaltsbuch & Budgetverwaltung")
        header.set_title_widget(title)

        # Menü
        menu = Gio.Menu()
        menu.append("CSV exportieren…", "win.export")
        menu.append("CSV importieren…", "win.import")
        menu.append("Monatsreport als PDF…", "win.reportpdf")
        menu.append("Backup erstellen…", "win.backup")
        menu.append("Backup wiederherstellen…", "win.restore")

        btn_menu = Gtk.MenuButton(icon_name="open-menu-symbolic")
        btn_menu.set_menu_model(menu)

        # Theme-Schalter
        self.btn_theme = Gtk.Button(label="Design")
        self.btn_theme.connect("clicked", self.toggle_theme)

        header.pack_end(btn_menu)
        header.pack_end(self.btn_theme)

        act_export = Gio.SimpleAction.new("export", None)
        act_import = Gio.SimpleAction.new("import", None)
        act_report = Gio.SimpleAction.new("reportpdf", None)
        act_backup = Gio.SimpleAction.new("backup", None)
        act_restore = Gio.SimpleAction.new("restore", None)

        act_export.connect("activate", self.on_export_csv)
        act_import.connect("activate", self.on_import_csv)
        act_report.connect("activate", self.on_export_month_pdf)
        act_backup.connect("activate", self.on_create_backup)
        act_restore.connect("activate", self.on_restore_backup)

        self.add_action(act_export)
        self.add_action(act_import)
        self.add_action(act_report)
        self.add_action(act_backup)
        self.add_action(act_restore)

        # NavigationSplitView
        split = Adw.NavigationSplitView()
        self.toolbar_view.set_content(split)

        # Sidebar
        sidebar_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        sidebar_box.set_margin_top(12)
        sidebar_box.set_margin_bottom(12)
        sidebar_box.set_margin_start(12)
        sidebar_box.set_margin_end(12)

        sidebar_page = Adw.NavigationPage(title="Menü", child=sidebar_box)
        split.set_sidebar(sidebar_page)

        lbl_title = Gtk.Label(label="FinanzPlaner")
        lbl_title.add_css_class("title-1")
        lbl_title.set_xalign(0)
        sidebar_box.append(lbl_title)

        lbl_sub = Gtk.Label(label="Dein modernes Haushaltsbuch")
        lbl_sub.add_css_class("dim-label")
        lbl_sub.set_xalign(0)
        sidebar_box.append(lbl_sub)

        sidebar_box.append(Gtk.Separator())

        self.btn_overview = Gtk.ToggleButton(label="Übersicht")
        self.btn_transactions = Gtk.ToggleButton(label="Transaktionen")
        self.btn_budgets = Gtk.ToggleButton(label="Budgets")

        for b in (self.btn_overview, self.btn_transactions, self.btn_budgets):
            b.add_css_class("flat")
            b.set_halign(Gtk.Align.FILL)
            sidebar_box.append(b)

        # Navigation → Handler
        self.btn_overview.connect("toggled", self._on_nav_toggled, "overview")
        self.btn_transactions.connect("toggled", self._on_nav_toggled, "transactions")
        self.btn_budgets.connect("toggled", self._on_nav_toggled, "budgets")

        # Seitenbereich
        content_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        content_box.set_margin_top(12)
        content_box.set_margin_bottom(12)
        content_box.set_margin_start(8)
        content_box.set_margin_end(12)

        content_page = Adw.NavigationPage(title="FinanzPlaner", child=content_box)
        split.set_content(content_page)

        # Stack
        self.stack = Gtk.Stack()
        self.stack.set_transition_type(Gtk.StackTransitionType.CROSSFADE)
        content_box.append(self.stack)

        # Seiten erzeugen
        self.page_overview = self._build_overview_page()
        self.page_transactions = self._build_transactions_page()
        self.page_budgets = self._build_budget_page()

        self.stack.add_named(self.page_overview, "overview")
        self.stack.add_named(self.page_transactions, "transactions")
        self.stack.add_named(self.page_budgets, "budgets")

        # Default Seite aktivieren
        self.switch_page("overview")

    # ╔══════════════════════════════╗
    #   Navigation Logik
    # ╚══════════════════════════════╝

    def _on_nav_toggled(self, btn, page_name):
        if self._is_navigating:
            return
        if btn.get_active():
            self.switch_page(page_name)

    def switch_page(self, name):
        self._is_navigating = True
        self.btn_overview.set_active(name == "overview")
        self.btn_transactions.set_active(name == "transactions")
        self.btn_budgets.set_active(name == "budgets")
        self.stack.set_visible_child_name(name)
        self._is_navigating = False

    # ╔══════════════════════════════╗
    #   Übersicht
    # ╚══════════════════════════════╝
    def _build_overview_page(self):
        page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=16)

        # Summary Cards
        box_sum = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
        page.append(box_sum)

        self.lbl_income = self._create_summary(box_sum, "Einnahmen")
        self.lbl_expense = self._create_summary(box_sum, "Ausgaben")
        self.lbl_balance = self._create_summary(box_sum, "Saldo")

        # Analyse-Button
        btn_analysis = Gtk.Button(label="Finanzanalyse")
        btn_analysis.add_css_class("flat")
        btn_analysis.set_halign(Gtk.Align.START)
        btn_analysis.connect("clicked", self.on_show_analysis)
        page.append(btn_analysis)

        # ────────────────────────────────────────────────────────
        # Charts Area
        # ────────────────────────────────────────────────────────
        charts_container = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=20)
        charts_container.set_margin_top(20)
        charts_container.set_homogeneous(True)  # Beide Charts gleich breit
        page.append(charts_container)

        # --- LINKER CHART (Donut) ---
        box_donut = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)

        lbl_donut = Gtk.Label(label="Ausgaben nach Kategorie")
        lbl_donut.add_css_class("heading")  # Fett gedruckt
        lbl_donut.set_xalign(0.5)  # Zentriert
        box_donut.append(lbl_donut)

        self.donut = DonutChartWidget({})
        self.donut.on_segment_clicked = self.on_category_clicked
        # Wir geben dem Donut etwas mehr Breite für die Legende
        self.donut.set_content_width(450)
        self.donut.set_content_height(250)
        box_donut.append(self.donut)

        charts_container.append(box_donut)

        # --- RECHTER CHART (Bars) ---
        box_bars = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)

        lbl_bars = Gtk.Label(label="Verlauf (Monate)")
        lbl_bars.add_css_class("heading")
        lbl_bars.set_xalign(0.5)
        box_bars.append(lbl_bars)

        self.bars = BarChartWidget({})
        self.bars.on_bar_clicked = self.on_month_clicked
        self.bars.set_content_height(250)
        box_bars.append(self.bars)

        charts_container.append(box_bars)
        # ────────────────────────────────────────────────────────

        # Filterbanner
        self.filter_banner = FilterBannerWidget()
        self.filter_banner.btn_clear.connect("clicked", lambda b: self.clear_filters())
        page.append(self.filter_banner)

        # Letzte Transaktionen
        lbl = Gtk.Label(label="Letzte Transaktionen")
        lbl.add_css_class("section-title")
        lbl.set_xalign(0)
        page.append(lbl)

        self.overview_list = TransactionListWidget(
            self,
            on_edit=self.edit_transaction,
            on_delete=self.delete_transaction,
        )

        sc = Gtk.ScrolledWindow()
        sc.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
        sc.set_child(self.overview_list)
        sc.set_min_content_height(200)
        page.append(sc)

        # Top 5 Kategorien
        lbl_top = Gtk.Label(label="Top 5 Kategorien")
        lbl_top.add_css_class("section-title")
        lbl_top.set_xalign(0)
        page.append(lbl_top)

        self.top5_listbox = Gtk.ListBox()
        self.top5_listbox.set_selection_mode(Gtk.SelectionMode.NONE)
        sc_top = Gtk.ScrolledWindow()
        sc_top.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
        sc_top.set_child(self.top5_listbox)
        sc_top.set_min_content_height(140)
        page.append(sc_top)

        # Größte Ausgaben
        lbl_big = Gtk.Label(label="Größte Ausgaben")
        lbl_big.add_css_class("section-title")
        lbl_big.set_xalign(0)
        page.append(lbl_big)

        self.biggest_listbox = Gtk.ListBox()
        self.biggest_listbox.set_selection_mode(Gtk.SelectionMode.NONE)
        sc_big = Gtk.ScrolledWindow()
        sc_big.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
        sc_big.set_child(self.biggest_listbox)
        sc_big.set_min_content_height(180)
        page.append(sc_big)

        return page

    # -----------------------------------
    # Summary Card Helper
    # -----------------------------------
    def _create_summary(self, parent, title):
        card = Adw.Bin()
        card.add_css_class("card")
        parent.append(card)

        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=4)
        box.set_margin_top(10)
        box.set_margin_bottom(10)
        box.set_margin_start(16)
        box.set_margin_end(16)
        card.set_child(box)

        lbl_t = Gtk.Label(label=title)
        lbl_t.add_css_class("summary-card-title")
        lbl_t.set_xalign(0)
        box.append(lbl_t)

        lbl_v = Gtk.Label(label="0,00 €")
        lbl_v.add_css_class("summary-card-value")
        lbl_v.set_xalign(0)
        box.append(lbl_v)

        return lbl_v

    # ╔══════════════════════════════╗
    #   Transaktionen Seite
    # ╚══════════════════════════════╝
    def _build_transactions_page(self):
        page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=16)

        # Form (Hinzufügen)
        form_card = Adw.Bin()
        form_card.add_css_class("card")
        page.append(form_card)

        form = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        form.set_margin_top(12)
        form.set_margin_bottom(12)
        form.set_margin_start(16)
        form.set_margin_end(16)
        form_card.set_child(form)

        lbl = Gtk.Label(label="Transaktion hinzufügen")
        lbl.add_css_class("section-title")
        lbl.set_xalign(0)
        form.append(lbl)

        grid = Gtk.Grid(column_spacing=12, row_spacing=12)
        form.append(grid)

        # Datum
        lbl_d = Gtk.Label(label="Datum")
        lbl_d.set_xalign(0)
        grid.attach(lbl_d, 0, 0, 1, 1)

        self.entry_date = Gtk.Entry(text=date.today().strftime("%d.%m.%Y"))
        grid.attach(self.entry_date, 0, 1, 1, 1)

        # Beschreibung
        lbl_desc = Gtk.Label(label="Beschreibung")
        lbl_desc.set_xalign(0)
        grid.attach(lbl_desc, 1, 0, 1, 1)

        self.entry_desc = Gtk.Entry()
        self.entry_desc.set_placeholder_text("z.B. Supermarkt")
        grid.attach(self.entry_desc, 1, 1, 1, 1)

        # Betrag
        lbl_amt = Gtk.Label(label="Betrag (€)")
        lbl_amt.set_xalign(0)
        grid.attach(lbl_amt, 2, 0, 1, 1)

        self.entry_amount = Gtk.Entry(text="0.00")
        grid.attach(self.entry_amount, 2, 1, 1, 1)

        # Typ
        lbl_type = Gtk.Label(label="Typ")
        lbl_type.set_xalign(0)
        grid.attach(lbl_type, 3, 0, 1, 1)

        self.combo_type = Gtk.ComboBoxText()
        self.combo_type.append_text("Ausgabe")
        self.combo_type.append_text("Einnahme")
        self.combo_type.set_active(0)
        grid.attach(self.combo_type, 3, 1, 1, 1)

        # Kategorie
        lbl_c = Gtk.Label(label="Kategorie")
        lbl_c.set_xalign(0)
        grid.attach(lbl_c, 0, 2, 1, 1)

        self.entry_category = Gtk.Entry()
        self.entry_category.set_placeholder_text("z.B. Lebensmittel")
        grid.attach(self.entry_category, 0, 3, 2, 1)

        # Autovervollständigung für Kategorien
        completion = Gtk.EntryCompletion()
        completion.set_model(self.category_store)
        completion.set_text_column(0)
        self.entry_category.set_completion(completion)

        # Add Button
        btn_add = Gtk.Button(label="+ Hinzufügen")
        btn_add.add_css_class("suggested-action")
        btn_add.connect("clicked", self.on_add_transaction)
        grid.attach(btn_add, 3, 3, 1, 1)

        # Schnellkategorie-Buttons
        quick_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        quick_box.set_margin_top(4)
        form.append(quick_box)

        quick_categories = ["Lebensmittel", "Tanken", "Freizeit", "Haushalt", "Sonstiges"]
        for qc in quick_categories:
            btn_q = Gtk.Button(label=qc)
            btn_q.add_css_class("quick-cat")
            btn_q.connect("clicked", lambda b, name=qc: self.entry_category.set_text(name))
            quick_box.append(btn_q)

        # ────────────────────────────────────────────────────────
        # NEU: Filter Leiste
        # ────────────────────────────────────────────────────────
        filter_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
        filter_box.set_margin_top(10)
        page.append(filter_box)

        lbl_filter = Gtk.Label(label="Filtern:")
        lbl_filter.add_css_class("heading")
        filter_box.append(lbl_filter)

        # Jahr Combo
        self.combo_filter_year = Gtk.ComboBoxText()
        filter_box.append(self.combo_filter_year)

        # Monat Combo
        self.combo_filter_month = Gtk.ComboBoxText()
        for i, m_name in enumerate(MONTH_NAMES):
            # ID="01", "02", ... Text="Januar"
            self.combo_filter_month.append(f"{i+1:02d}", m_name)
        filter_box.append(self.combo_filter_month)

        # Reset
        btn_reset = Gtk.Button(label="Zurücksetzen")
        btn_reset.connect("clicked", lambda b: self.clear_filters())
        filter_box.append(btn_reset)

        # Initial befüllen
        self._init_year_combo()

        # Signale verbinden
        self.combo_filter_year.connect("changed", self._on_filter_combo_changed)
        self.combo_filter_month.connect("changed", self._on_filter_combo_changed)
        # ────────────────────────────────────────────────────────

        # Tabelle Title
        lbl2 = Gtk.Label(label="Alle Transaktionen")
        lbl2.add_css_class("section-title")
        lbl2.set_xalign(0)
        page.append(lbl2)

        self.tx_list = TransactionListWidget(
            self,
            on_edit=self.edit_transaction,
            on_delete=self.delete_transaction,
        )

        # Doppelklick -> Bearbeiten, KEIN Filter mehr
        self.tx_list.connect("row-activated", self.on_tx_row_activated)

        sc = Gtk.ScrolledWindow()
        sc.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
        sc.set_child(self.tx_list)
        sc.set_min_content_height(350)
        page.append(sc)

        return page

    # ╔══════════════════════════════╗
    #   Budget Seite (inkl. wiederkehrende Buchungen)
    # ╚══════════════════════════════╝
    def _build_budget_page(self):
        page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=16)

        lbl = Gtk.Label(label="Budgetübersicht")
        lbl.add_css_class("section-title")
        lbl.set_xalign(0)
        page.append(lbl)

        self.budget_listbox = Gtk.ListBox()
        self.budget_listbox.set_selection_mode(Gtk.SelectionMode.NONE)
        page.append(self.budget_listbox)

        # Form Budget
        form_card = Adw.Bin()
        form_card.add_css_class("card")
        page.append(form_card)

        form = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)
        form.set_margin_top(12)
        form.set_margin_bottom(12)
        form.set_margin_start(16)
        form.set_margin_end(16)
        form_card.set_child(form)

        header = Gtk.Label(label="Budget hinzufügen / ändern")
        header.add_css_class("section-title")
        header.set_xalign(0)
        form.append(header)

        grid = Gtk.Grid(column_spacing=12, row_spacing=12)
        form.append(grid)

        lbl_cat = Gtk.Label(label="Kategorie")
        lbl_cat.set_xalign(0)
        grid.attach(lbl_cat, 0, 0, 1, 1)

        self.entry_budget_cat = Gtk.Entry()
        self.entry_budget_cat.set_placeholder_text("z.B. Lebensmittel")
        grid.attach(self.entry_budget_cat, 0, 1, 1, 1)

        lbl_lim = Gtk.Label(label="Monatslimit (€)")
        lbl_lim.set_xalign(0)
        grid.attach(lbl_lim, 1, 0, 1, 1)

        self.entry_budget_limit = Gtk.Entry()
        self.entry_budget_limit.set_placeholder_text("z.B. 300")
        grid.attach(self.entry_budget_limit, 1, 1, 1, 1)

        btn_save = Gtk.Button(label="Budget speichern")
        btn_save.add_css_class("suggested-action")
        btn_save.connect("clicked", self.on_save_budget)
        grid.attach(btn_save, 2, 1, 1, 1)

        # ────────────────────────────────────────────────────────
        # Wiederkehrende Buchungen (Abos etc.)
        # ────────────────────────────────────────────────────────
        sep = Gtk.Separator()
        page.append(sep)

        lbl_rec = Gtk.Label(label="Wiederkehrende Buchungen (Miete, Strom, Abos …)")
        lbl_rec.add_css_class("section-title")
        lbl_rec.set_xalign(0)
        page.append(lbl_rec)

        self.recurring_listbox = Gtk.ListBox()
        self.recurring_listbox.set_selection_mode(Gtk.SelectionMode.NONE)
        self.recurring_listbox.connect("row-activated", self.on_recurring_row_activated)
        sc_rec = Gtk.ScrolledWindow()
        sc_rec.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
        sc_rec.set_child(self.recurring_listbox)
        sc_rec.set_min_content_height(180)
        page.append(sc_rec)

        form_rec_card = Adw.Bin()
        form_rec_card.add_css_class("card")
        page.append(form_rec_card)

        form_rec = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)
        form_rec.set_margin_top(12)
        form_rec.set_margin_bottom(12)
        form_rec.set_margin_start(16)
        form_rec.set_margin_end(16)
        form_rec_card.set_child(form_rec)

        header_rec = Gtk.Label(label="Wiederkehrende Buchung hinzufügen")
        header_rec.add_css_class("section-title")
        header_rec.set_xalign(0)
        form_rec.append(header_rec)

        grid2 = Gtk.Grid(column_spacing=12, row_spacing=8)
        form_rec.append(grid2)

        # Beschreibung
        lbl_rd = Gtk.Label(label="Beschreibung")
        lbl_rd.set_xalign(0)
        grid2.attach(lbl_rd, 0, 0, 1, 1)
        self.entry_rec_desc = Gtk.Entry()
        self.entry_rec_desc.set_placeholder_text("z.B. Miete Wohnung")
        grid2.attach(self.entry_rec_desc, 0, 1, 1, 1)

        # Kategorie
        lbl_rc = Gtk.Label(label="Kategorie")
        lbl_rc.set_xalign(0)
        grid2.attach(lbl_rc, 1, 0, 1, 1)
        self.entry_rec_cat = Gtk.Entry()
        self.entry_rec_cat.set_placeholder_text("z.B. Wohnen")
        grid2.attach(self.entry_rec_cat, 1, 1, 1, 1)

        # Betrag
        lbl_ra = Gtk.Label(label="Betrag (€)")
        lbl_ra.set_xalign(0)
        grid2.attach(lbl_ra, 2, 0, 1, 1)
        self.entry_rec_amount = Gtk.Entry()
        self.entry_rec_amount.set_placeholder_text("z.B. 800.00")
        grid2.attach(self.entry_rec_amount, 2, 1, 1, 1)

        # Typ
        lbl_rt = Gtk.Label(label="Typ")
        lbl_rt.set_xalign(0)
        grid2.attach(lbl_rt, 3, 0, 1, 1)
        self.combo_rec_type = Gtk.ComboBoxText()
        self.combo_rec_type.append_text("Ausgabe")
        self.combo_rec_type.append_text("Einnahme")
        self.combo_rec_type.set_active(0)
        grid2.attach(self.combo_rec_type, 3, 1, 1, 1)

        # Intervall
        lbl_ri = Gtk.Label(label="Intervall (Monate)")
        lbl_ri.set_xalign(0)
        grid2.attach(lbl_ri, 0, 2, 1, 1)
        self.entry_rec_interval = Gtk.Entry()
        self.entry_rec_interval.set_placeholder_text("z.B. 1")
        grid2.attach(self.entry_rec_interval, 0, 3, 1, 1)

        # Startdatum
        lbl_rs = Gtk.Label(label="Startdatum (TT.MM.JJJJ oder JJJJ-MM-TT, leer = heute)")
        lbl_rs.set_xalign(0)
        grid2.attach(lbl_rs, 1, 2, 2, 1)
        self.entry_rec_startdate = Gtk.Entry()
        grid2.attach(self.entry_rec_startdate, 1, 3, 2, 1)

        # Button
        btn_rec_save = Gtk.Button(label="Abo speichern")
        btn_rec_save.add_css_class("suggested-action")
        btn_rec_save.connect("clicked", self.on_save_recurring)
        grid2.attach(btn_rec_save, 3, 3, 1, 1)

        return page

    # ╔══════════════════════════════╗
    #   TX-Row-Handler
    # ╚══════════════════════════════╝
    def on_tx_row_activated(self, listbox, row):
        """
        Doppelklick auf eine Transaktion -> Bearbeiten-Dialog öffnen.
        KEINE automatische Filterung mehr.
        """
        tx = getattr(row, "tx", None)
        if tx:
            self.edit_transaction(tx)

    # ╔══════════════════════════════╗
    #   Filter-Logik & Dropdowns
    # ╚══════════════════════════════╝

    def _init_year_combo(self):
        """Lädt alle verfügbaren Jahre in das Dropdown."""
        self._updating_filter_combo = True
        self.combo_filter_year.remove_all()

        # Hole alle Jahre aus der DB
        all_tx = self.db.get_all_transactions()
        years = set()
        for t in all_tx:
            dt = parse_date(t.date)
            if dt:
                years.add(dt.year)

        # Füge aktuelles Jahr hinzu, falls keine Daten
        if not years:
            years.add(date.today().year)

        for y in sorted(years, reverse=True):
            self.combo_filter_year.append(str(y), str(y))

        self._updating_filter_combo = False

    def _on_filter_combo_changed(self, combo):
        """Wird aufgerufen, wenn User Jahr oder Monat ändert."""
        if self._updating_filter_combo:
            return

        y_str = self.combo_filter_year.get_active_id()
        m_str = self.combo_filter_month.get_active_id()

        if y_str and m_str:
            # Beides gewählt -> Filter setzen
            year = int(y_str)
            month = int(m_str)
            self.active_category_filter = None
            self.active_month_filter = (year, month)
            self.refresh_all()
        else:
            # Nur eines gewählt oder keines -> noch nicht filtern
            pass

    def clear_filters(self, *_):
        self._updating_filter_combo = True
        self.combo_filter_year.set_active_id(None)
        self.combo_filter_month.set_active_id(None)
        self._updating_filter_combo = False

        self.active_category_filter = None
        self.active_month_filter = None
        self.refresh_all()

    def on_category_clicked(self, category):
        # Wenn Kategorie in Charts geklickt wird
        self.active_month_filter = None
        self.active_category_filter = category
        self.refresh_all()

    def on_month_clicked(self, year, month):
        # Wenn Balken in Charts geklickt wird
        self.active_category_filter = None
        self.active_month_filter = (year, month)
        self.refresh_all()

    # ╔══════════════════════════════╗
    #   Refresh / Charts / Summary
    # ╚══════════════════════════════╝
    def refresh_all(self):
        # Wir laden immer alle, um Jahre fürs Dropdown zu finden
        all_tx = self.db.get_all_transactions()

        # Falls neue Jahre dazugekommen sind (durch CSV Import etc.), aktualisieren
        current_y_id = self.combo_filter_year.get_active_id()
        self._init_year_combo()
        if current_y_id:
            self._updating_filter_combo = True
            self.combo_filter_year.set_active_id(current_y_id)
            self._updating_filter_combo = False

        # Filter anwenden
        if self.active_category_filter:
            self.transactions = [t for t in all_tx if t.category == self.active_category_filter]
            self.filter_banner.show_category_filter(self.active_category_filter, self.clear_filters)

            # Dropdowns leeren, da Kategorie-Filter aktiv
            self._updating_filter_combo = True
            self.combo_filter_year.set_active_id(None)
            self.combo_filter_month.set_active_id(None)
            self._updating_filter_combo = False

        elif self.active_month_filter:
            y, m = self.active_month_filter
            self.transactions = []
            for t in all_tx:
                dt = parse_date(t.date)
                if dt and dt.year == y and dt.month == m:
                    self.transactions.append(t)
            self.filter_banner.show_month_filter(y, m, self.clear_filters)

            # Dropdowns synchronisieren (falls Filter durch Chart gesetzt wurde)
            self._updating_filter_combo = True
            self.combo_filter_year.set_active_id(str(y))
            self.combo_filter_month.set_active_id(f"{m:02d}")
            self._updating_filter_combo = False

        else:
            self.transactions = all_tx
            self.filter_banner.hide_filter()

        self.budgets = self.db.get_budgets()

        self.update_summary()
        self.update_charts()
        self.update_budget_list()
        self.update_recurring_list()

        # Kategorienliste für Autocomplete aktualisieren
        self.category_store.clear()
        for t in self.db.get_all_transactions():
            self.category_store.append([t.category])

    def update_summary(self):
        income = sum(t.amount for t in self.transactions if t.is_income)
        expense = sum(t.amount for t in self.transactions if not t.is_income)
        self.lbl_income.set_text(format_euro(income))
        self.lbl_expense.set_text(format_euro(expense))
        self.lbl_balance.set_text(format_euro(income - expense))

    def update_charts(self):
        cat_sum = defaultdict(float)
        for t in self.transactions:
            if not t.is_income:
                cat_sum[t.category] += t.amount
        self.donut.update_data(cat_sum)

        month_sum = defaultdict(float)
        for t in self.transactions:
            dt = parse_date(t.date)
            if dt and not t.is_income:
                month_sum[month_key(dt)] += t.amount
        self.bars.update_data(month_sum)

        # Update Lists
        self.overview_list.update_transactions(self.transactions[-20:])
        self.tx_list.update_transactions(self.transactions)

        # Top 5 Kategorien (nur Ausgaben)
        top_items = sorted(cat_sum.items(), key=lambda x: x[1], reverse=True)[:5]
        self._update_top5_list(top_items)

        # Größte 10 Ausgaben
        largest = sorted(
            [t for t in self.transactions if not t.is_income],
            key=lambda t: t.amount,
            reverse=True
        )[:10]
        self._update_biggest_list(largest)

    def _update_top5_list(self, items):
        # GTK4 ListBox leeren
        child = self.top5_listbox.get_first_child()
        while child:
            nxt = child.get_next_sibling()
            self.top5_listbox.remove(child)
            child = nxt

        for cat, val in items:
            row = Gtk.ListBoxRow()
            box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
            box.set_margin_top(3)
            box.set_margin_bottom(3)
            box.set_margin_start(8)
            box.set_margin_end(8)
            row.set_child(box)

            lbl_cat = Gtk.Label(label=cat)
            lbl_cat.set_xalign(0)
            lbl_cat.set_hexpand(True)
            box.append(lbl_cat)

            lbl_val = Gtk.Label(label=format_euro(val))
            lbl_val.set_xalign(1)
            box.append(lbl_val)

            self.top5_listbox.append(row)

    def _update_biggest_list(self, transactions):
        child = self.biggest_listbox.get_first_child()
        while child:
            nxt = child.get_next_sibling()
            self.biggest_listbox.remove(child)
            child = nxt

        for t in transactions:
            row = Gtk.ListBoxRow()
            box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
            box.set_margin_top(3)
            box.set_margin_bottom(3)
            box.set_margin_start(8)
            box.set_margin_end(8)
            row.set_child(box)

            lbl_date = Gtk.Label(label=t.date)
            lbl_date.set_xalign(0)
            box.append(lbl_date)

            lbl_desc = Gtk.Label(label=t.description)
            lbl_desc.set_xalign(0)
            lbl_desc.set_hexpand(True)
            lbl_desc.set_ellipsize(3)
            box.append(lbl_desc)

            lbl_cat = Gtk.Label(label=t.category)
            lbl_cat.set_xalign(0)
            lbl_cat.set_hexpand(True)
            lbl_cat.set_ellipsize(3)
            box.append(lbl_cat)

            lbl_val = Gtk.Label(label=format_euro(t.amount))
            lbl_val.set_xalign(1)
            lbl_val.add_css_class("amount-expense")
            box.append(lbl_val)

            self.biggest_listbox.append(row)

    def update_budget_list(self):
        # Clear GTK4 ListBox
        child = self.budget_listbox.get_first_child()
        while child:
            nxt = child.get_next_sibling()
            self.budget_listbox.remove(child)
            child = nxt

        spending = defaultdict(float)

        # Budget Logik:
        # Falls Filter aktiv -> berechne Budget für den gefilterten Monat
        # Falls kein Filter -> berechne für aktuellen Monat
        if self.active_month_filter:
            relevant_tx = self.transactions
        else:
            y_now, m_now = date.today().year, date.today().month
            for t in self.db.get_all_transactions():
                dt = parse_date(t.date)
                if dt and dt.year == y_now and dt.month == m_now and not t.is_income:
                    spending[t.category] += t.amount
            relevant_tx = []  # schon abgearbeitet

        if self.active_month_filter:
            for t in relevant_tx:
                if not t.is_income:
                    spending[t.category] += t.amount

        for cat, budget in sorted(self.budgets.items()):
            spent = spending.get(cat, 0.0)

            row = Gtk.ListBoxRow()
            box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
            box.set_margin_top(4)
            box.set_margin_bottom(4)
            box.set_margin_start(10)
            box.set_margin_end(10)
            row.set_child(box)

            lbl_cat = Gtk.Label(label=cat)
            lbl_cat.set_xalign(0)
            lbl_cat.set_hexpand(True)
            box.append(lbl_cat)

            # Verhältnis
            if budget.monthly_limit > 0:
                ratio = spent / budget.monthly_limit
            else:
                ratio = 0.0

            lbl_val = Gtk.Label(label=f"{format_euro(spent)} / {format_euro(budget.monthly_limit)}")
            lbl_val.set_xalign(1)
            box.append(lbl_val)

            status = Gtk.Label()
            if budget.monthly_limit <= 0:
                status.set_text("Kein Limit")
            else:
                if ratio < 0.6:
                    status.set_text("OK")
                    status.add_css_class("budget-ok")
                elif ratio <= 1.0:
                    status.set_text("Warnung")
                    status.add_css_class("budget-warn")
                else:
                    status.set_text("Überschritten")
                    status.add_css_class("budget-over")

            status.set_xalign(1)
            box.append(status)

            self.budget_listbox.append(row)

    def update_recurring_list(self):
        # Liste der wiederkehrenden Buchungen aktualisieren
        child = self.recurring_listbox.get_first_child()
        while child:
            nxt = child.get_next_sibling()
            self.recurring_listbox.remove(child)
            child = nxt

        recurring = self.db.get_recurring()
        for rt in recurring:
            row = Gtk.ListBoxRow()
            row.rt = rt
            box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
            box.set_margin_top(3)
            box.set_margin_bottom(3)
            box.set_margin_start(8)
            box.set_margin_end(8)
            row.set_child(box)

            lbl_desc = Gtk.Label(label=rt.description)
            lbl_desc.set_xalign(0)
            lbl_desc.set_hexpand(True)
            box.append(lbl_desc)

            lbl_cat = Gtk.Label(label=rt.category)
            lbl_cat.set_xalign(0)
            lbl_cat.set_hexpand(True)
            box.append(lbl_cat)

            lbl_amt = Gtk.Label(label=format_euro(rt.amount))
            lbl_amt.set_xalign(1)
            if rt.is_income:
                lbl_amt.add_css_class("amount-income")
            else:
                lbl_amt.add_css_class("amount-expense")
            box.append(lbl_amt)

            lbl_int = Gtk.Label(label=f"alle {rt.interval_months} Monate")
            lbl_int.set_xalign(1)
            box.append(lbl_int)

            lbl_next = Gtk.Label(label=f"nächste Buchung: {rt.next_date}")
            lbl_next.set_xalign(1)
            box.append(lbl_next)

            self.recurring_listbox.append(row)

    # ╔══════════════════════════════╗
    #   Transaktions-CRUD
    # ╚══════════════════════════════╝
    def on_add_transaction(self, *_):
        try:
            amount = float(self.entry_amount.get_text().replace(",", "."))
        except Exception:
            self.show_error("Ungültiger Betrag.")
            return

        tx = Transaction(
            id=None,
            date=self.entry_date.get_text(),
            description=self.entry_desc.get_text() or "(leer)",
            category=self.entry_category.get_text() or "(Unbekannt)",
            amount=amount,
            is_income=(self.combo_type.get_active_text() == "Einnahme"),
        )

        tx.id = self.db.add_transaction(tx)

        self.entry_amount.set_text("0.00")
        self.entry_desc.set_text("")
        self.entry_category.set_text("")

        self.refresh_all()

    def edit_transaction(self, tx):
        dlg = EditTransactionDialog(self, tx, on_save_callback=self._save_tx)
        dlg.show()

    def _save_tx(self, tx):
        self.db.update_transaction(tx)
        self.refresh_all()

    def delete_transaction(self, tx):
        dlg = DeleteConfirmationDialog(self, tx, on_delete_callback=self._delete_tx)
        dlg.show()

    def _delete_tx(self, tx):
        if tx.id:
            self.db.delete_transaction(tx.id)
        self.refresh_all()

    # ╔══════════════════════════════╗
    #   Wiederkehrende Buchungen: CRUD
    # ╚══════════════════════════════╝
    def on_save_recurring(self, *_):
        desc = self.entry_rec_desc.get_text().strip() or "(leer)"
        cat = self.entry_rec_cat.get_text().strip() or "(Unbekannt)"

        try:
            amount = float(self.entry_rec_amount.get_text().replace(",", "."))
        except Exception:
            self.show_error("Ungültiger Betrag für wiederkehrende Buchung.")
            return

        try:
            interval = int(self.entry_rec_interval.get_text().strip())
        except Exception:
            self.show_error("Ungültiges Intervall (Monate).")
            return

        start_raw = self.entry_rec_startdate.get_text().strip()
        if start_raw:
            dt = parse_date(start_raw)
            if not dt:
                self.show_error("Ungültiges Startdatum.")
                return
            next_date_str = dt.strftime("%Y-%m-%d")
        else:
            next_date_str = date.today().strftime("%Y-%m-%d")

        rt = RecurringTransaction(
            id=None,
            description=desc,
            category=cat,
            amount=amount,
            is_income=(self.combo_rec_type.get_active_text() == "Einnahme"),
            interval_months=interval,
            next_date=next_date_str,
        )
        self.db.add_recurring(rt)

        # Felder leeren
        self.entry_rec_desc.set_text("")
        self.entry_rec_cat.set_text("")
        self.entry_rec_amount.set_text("")
        self.entry_rec_interval.set_text("")
        self.entry_rec_startdate.set_text("")

        self.refresh_all()

    def on_recurring_row_activated(self, listbox, row):
        rt = getattr(row, "rt", None)
        if not rt or rt.id is None:
            return

        dlg = Adw.MessageDialog(
            transient_for=self,
            heading="Wiederkehrende Buchung löschen?",
            body=f"Soll die wiederkehrende Buchung\n\n{rt.description}\n\nwirklich gelöscht werden?"
        )
        dlg.add_response("cancel", "Abbrechen")
        dlg.add_response("delete", "Löschen")
        dlg.set_default_response("delete")

        def _on_resp(d, resp):
            if resp == "delete":
                self.db.delete_recurring(rt.id)
                self.refresh_all()
            d.destroy()

        dlg.connect("response", _on_resp)
        dlg.show()

    # ╔══════════════════════════════╗
    #   Budget speichern
    # ╚══════════════════════════════╝
    def on_save_budget(self, *_):
        cat = self.entry_budget_cat.get_text().strip()
        try:
            limit = float(self.entry_budget_limit.get_text().replace(",", "."))
        except Exception:
            self.show_error("Ungültiges Limit.")
            return

        if not cat:
            self.show_error("Bitte Kategorie eingeben.")
            return

        self.db.upsert_budget(cat, limit)

        self.entry_budget_cat.set_text("")
        self.entry_budget_limit.set_text("")

        self.refresh_all()

    # ╔══════════════════════════════╗
    #   CSV Import / Export
    # ╚══════════════════════════════╝
    def on_export_csv(self, *_):
        dlg = Gtk.FileDialog()
        dlg.set_initial_name("finanzplaner_export.csv")

        def after(dialog, result):
            try:
                fobj = dialog.save_finish(result)
            except Exception:
                return

            path = fobj.get_path()
            if not path:
                return

            with open(path, "w", newline="", encoding="utf-8") as f:
                w = csv.writer(f, delimiter=";")
                w.writerow(["date", "description", "category", "amount", "is_income"])
                for t in self.db.get_all_transactions():
                    w.writerow([t.date, t.description, t.category, t.amount, int(t.is_income)])

        dlg.save(self, None, after)

    def on_import_csv(self, *_):
        dlg = Gtk.FileDialog()

        def after(dialog, result):
            try:
                fobj = dialog.open_finish(result)
            except Exception:
                return

            path = fobj.get_path()
            if not path:
                return

            with open(path, "r", newline="", encoding="utf-8") as f:
                r = csv.DictReader(f, delimiter=";")
                for row in r:
                    try:
                        amount = float(row["amount"])
                        is_income = bool(int(row["is_income"]))
                    except Exception:
                        continue

                    tx = Transaction(
                        id=None,
                        date=row["date"],
                        description=row["description"],
                        category=row["category"],
                        amount=amount,
                        is_income=is_income,
                    )
                    self.db.add_transaction(tx)

            self.refresh_all()

        dlg.open(self, None, after)

    # ╔══════════════════════════════╗
    #   Monatsreport als PDF
    # ╚══════════════════════════════╝
    def on_export_month_pdf(self, *_):
        if pdfcanvas is None:
            self.show_error(
                "Das Modul 'reportlab' ist nicht installiert.\n"
                "Bitte installiere es mit:\n\npip install reportlab"
            )
            return

        # Monat bestimmen
        all_tx = self.db.get_all_transactions()
        month_expenses = defaultdict(float)

        for t in all_tx:
            dt = parse_date(t.date)
            if dt and not t.is_income:
                key = (dt.year, dt.month)
                month_expenses[key] += t.amount

        if self.active_month_filter:
            year, month = self.active_month_filter
        else:
            if month_expenses:
                year, month = sorted(month_expenses.keys())[-1]
            else:
                today = date.today()
                year, month = today.year, today.month

        # Transaktionen dieses Monats
        txs = []
        for t in all_tx:
            dt = parse_date(t.date)
            if dt and dt.year == year and dt.month == month:
                txs.append(t)

        dlg = Gtk.FileDialog()
        dlg.set_initial_name(f"Monatsreport_{year}-{month:02d}.pdf")

        def after(dialog, result):
            try:
                fobj = dialog.save_finish(result)
            except Exception:
                return

            path = fobj.get_path()
            if not path:
                return

            c = pdfcanvas.Canvas(path, pagesize=A4)
            width, height = A4

            y = height - 2 * cm
            c.setFont("Helvetica-Bold", 16)
            c.drawString(2 * cm, y, f"Monatsreport {MONTH_NAMES[month - 1]} {year}")
            y -= 1.5 * cm

            income = sum(t.amount for t in txs if t.is_income)
            expense = sum(t.amount for t in txs if not t.is_income)
            balance = income - expense

            c.setFont("Helvetica", 11)
            c.drawString(2 * cm, y, f"Einnahmen: {format_euro(income)}")
            y -= 0.7 * cm
            c.drawString(2 * cm, y, f"Ausgaben:  {format_euro(expense)}")
            y -= 0.7 * cm
            c.drawString(2 * cm, y, f"Saldo:     {format_euro(balance)}")
            y -= 1.2 * cm

            # Kategorien
            cat_sum = defaultdict(float)
            for t in txs:
                if not t.is_income:
                    cat_sum[t.category] += t.amount

            c.setFont("Helvetica-Bold", 12)
            c.drawString(2 * cm, y, "Ausgaben nach Kategorien:")
            y -= 0.8 * cm
            c.setFont("Helvetica", 10)
            for cat, val in sorted(cat_sum.items(), key=lambda x: x[1], reverse=True):
                if y < 3 * cm:
                    c.showPage()
                    y = height - 2 * cm
                    c.setFont("Helvetica", 10)
                c.drawString(2 * cm, y, f"- {cat}: {format_euro(val)}")
                y -= 0.6 * cm

            # Einzelne Transaktionen
            y -= 0.8 * cm
            if y < 3 * cm:
                c.showPage()
                y = height - 2 * cm
            c.setFont("Helvetica-Bold", 12)
            c.drawString(2 * cm, y, "Transaktionen:")
            y -= 0.8 * cm
            c.setFont("Helvetica", 9)

            for t in txs:
                line = f"{t.date} | {t.description} | {t.category} | "
                typ = "Einnahme" if t.is_income else "Ausgabe"
                line += f"{typ} | {format_euro(t.amount)}"
                if y < 2 * cm:
                    c.showPage()
                    y = height - 2 * cm
                    c.setFont("Helvetica", 9)
                c.drawString(2 * cm, y, line[:120])
                y -= 0.5 * cm

            c.showPage()
            c.save()

        dlg.save(self, None, after)

    # ╔══════════════════════════════╗
    #   Backup-Funktion
    # ╚══════════════════════════════╝
    def on_create_backup(self, *_):
        """
        Erstellt ein Backup der haushalt.db in einem frei wählbaren Ordner/Laufwerk.
        """
        dlg = Gtk.FileDialog()

        def after(dialog, result):
            try:
                folder = dialog.select_folder_finish(result)
            except Exception:
                return

            folder_path = folder.get_path()
            if not folder_path:
                return

            timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
            backup_name = f"finanzplaner_backup_{timestamp}.db"
            backup_path = f"{folder_path}/{backup_name}"

            try:
                shutil.copy("haushalt.db", backup_path)
            except Exception as e:
                self.show_error(f"Backup fehlgeschlagen:\n{e}")
                return

            dlg_ok = Adw.MessageDialog(
                transient_for=self,
                heading="Backup erfolgreich",
                body=f"Backup wurde gespeichert unter:\n{backup_path}",
            )
            dlg_ok.add_response("ok", "OK")
            dlg_ok.set_default_response("ok")
            dlg_ok.connect("response", lambda d, r: d.destroy())
            dlg_ok.show()

        dlg.select_folder(self, None, after)

    # ╔══════════════════════════════╗
    #   Restore-Funktion
    # ╚══════════════════════════════╝
    def on_restore_backup(self, *_):
        """
        Stellt ein zuvor erstelltes Backup (.db) als haushalt.db wieder her.
        Achtung: aktuelle Daten werden überschrieben.
        """
        dlg = Gtk.FileDialog()

        def after(dialog, result):
            try:
                fobj = dialog.open_finish(result)
            except Exception:
                return

            path = fobj.get_path()
            if not path:
                return

            # Sicherheitsabfrage
            warn = Adw.MessageDialog(
                transient_for=self,
                heading="Backup wiederherstellen?",
                body=(
                    "Die aktuelle Datenbank wird durch das gewählte Backup ersetzt.\n"
                    "Nicht gesicherte Änderungen gehen verloren.\n\n"
                    f"Backup-Datei:\n{path}"
                ),
            )
            warn.add_response("cancel", "Abbrechen")
            warn.add_response("restore", "Wiederherstellen")
            warn.set_default_response("restore")

            def _on_resp(d, resp):
                if resp != "restore":
                    d.destroy()
                    return

                # DB schließen, Datei kopieren, DB neu öffnen
                try:
                    self.db.close()
                except Exception:
                    pass

                try:
                    shutil.copy(path, "haushalt.db")
                except Exception as e:
                    self.show_error(f"Restore fehlgeschlagen:\n{e}")
                    d.destroy()
                    return

                # Neu initialisieren
                self.db = Database()
                self.db.apply_recurring(date.today())
                self.transactions = self.db.get_all_transactions()
                self.budgets = self.db.get_budgets()
                self.refresh_all()

                info = Adw.MessageDialog(
                    transient_for=self,
                    heading="Restore erfolgreich",
                    body="Das Backup wurde erfolgreich wiederhergestellt.",
                )
                info.add_response("ok", "OK")
                info.set_default_response("ok")
                info.connect("response", lambda dd, r: dd.destroy())
                info.show()

                d.destroy()

            warn.connect("response", _on_resp)
            warn.show()

        dlg.open(self, None, after)

    # ╔══════════════════════════════╗
    #   Finanzanalyse (Text-Auswertung)
    # ╚══════════════════════════════╝
    def on_show_analysis(self, *_):
        all_tx = self.db.get_all_transactions()
        if not all_tx:
            self.show_error("Noch keine Daten für eine Analyse vorhanden.")
            return

        month_expenses = defaultdict(float)
        for t in all_tx:
            dt = parse_date(t.date)
            if dt and not t.is_income:
                key = (dt.year, dt.month)
                month_expenses[key] += t.amount

        if not month_expenses:
            self.show_error("Es gibt noch keine Ausgaben für eine Analyse.")
            return

        # Monat der Analyse
        if self.active_month_filter:
            year, month = self.active_month_filter
        else:
            year, month = sorted(month_expenses.keys())[-1]

        current_total = month_expenses[(year, month)]

        # Durchschnitt der letzten bis zu 6 Monate
        keys_sorted = sorted(month_expenses.keys())
        last_keys = [k for k in keys_sorted if k <= (year, month)][-6:]
        if last_keys:
            avg = sum(month_expenses[k] for k in last_keys) / len(last_keys)
        else:
            avg = current_total

        if avg == 0:
            diff_percent = 0.0
        else:
            diff_percent = (current_total - avg) / avg * 100.0

        # Top-Kategorie im Analysemonat
        cat_sum_month = defaultdict(float)
        for t in all_tx:
            dt = parse_date(t.date)
            if dt and dt.year == year and dt.month == month and not t.is_income:
                cat_sum_month[t.category] += t.amount

        if cat_sum_month:
            top_cat, top_val = sorted(cat_sum_month.items(), key=lambda x: x[1], reverse=True)[0]
            top_share = (top_val / current_total * 100.0) if current_total > 0 else 0.0
        else:
            top_cat, top_val, top_share = "(keine)", 0.0, 0.0

        text_lines = []
        text_lines.append(f"Analyse für {MONTH_NAMES[month - 1]} {year}")
        text_lines.append("")
        text_lines.append(f"Gesamtausgaben: {format_euro(current_total)}")
        text_lines.append(f"Durchschnitt (letzte bis zu 6 Monate): {format_euro(avg)}")

        if diff_percent > 5:
            text_lines.append(f"Du gibst ca. {diff_percent:.1f}% mehr aus als im Durchschnitt.")
        elif diff_percent < -5:
            text_lines.append(f"Du gibst ca. {abs(diff_percent):.1f}% weniger aus als im Durchschnitt. Stark!")
        else:
            text_lines.append("Deine Ausgaben liegen ungefähr auf deinem Durchschnittsniveau.")

        text_lines.append("")
        text_lines.append(f"Top-Kategorie: {top_cat} ({format_euro(top_val)}, {top_share:.1f}% der Ausgaben)")
        text_lines.append("")
        text_lines.append("Tipp: Prüfe, ob du in dieser Kategorie Einsparpotenzial hast oder ein Budget setzen möchtest.")

        body = "\n".join(text_lines)

        dlg = Adw.MessageDialog(
            transient_for=self,
            heading="Finanzanalyse",
            body=body,
        )
        dlg.add_response("ok", "OK")
        dlg.set_default_response("ok")
        dlg.connect("response", lambda d, r: d.destroy())
        dlg.show()

    # ╔══════════════════════════════╗
    #   Theme-Schalter
    # ╚══════════════════════════════╝
    def toggle_theme(self, *_):
        sm = Adw.StyleManager.get_default()
        scheme = sm.get_color_scheme()
        if scheme == Adw.ColorScheme.DEFAULT:
            sm.set_color_scheme(Adw.ColorScheme.FORCE_DARK)
        elif scheme == Adw.ColorScheme.FORCE_DARK:
            sm.set_color_scheme(Adw.ColorScheme.FORCE_LIGHT)
        else:
            sm.set_color_scheme(Adw.ColorScheme.DEFAULT)

    # ╔══════════════════════════════╗
    #   Fehlermeldung
    # ╚══════════════════════════════╝
    def show_error(self, msg):
        dlg = Adw.MessageDialog(
            transient_for=self,
            heading="Fehler",
            body=msg,
        )
        dlg.add_response("ok", "OK")
        dlg.set_default_response("ok")
        dlg.connect("response", lambda d, r: d.destroy())
        dlg.show()


# ╔══════════════════════════════╗
#   APPLICATION
# ╚══════════════════════════════╝

class FinanzApp(Adw.Application):
    def __init__(self):
        super().__init__(application_id="de.finanzplaner.app")

    def do_activate(self, *_):
        win = MainWindow(self)
        win.present()


def main():
    Adw.init()
    app = FinanzApp()
    app.run([])


if __name__ == "__main__":
    main()
