#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
widgets.py
Reusable Widgets für FinanzPlaner:
- TransactionListWidget (Listbox + Menü)
- EditTransactionDialog
- DeleteConfirmationDialog
- FilterBannerWidget
"""

from __future__ import annotations
from gi.repository import Gtk, Adw, Gio

from db import Transaction
from utils import format_euro


# ╔══════════════════════════════════════════════════════════════╗
#   EDIT-DIALOG
# ╚══════════════════════════════════════════════════════════════╝

class EditTransactionDialog(Adw.MessageDialog):
    """
    Dialog zum Bearbeiten einer Transaktion.
    """

    def __init__(self, parent, tx: Transaction, on_save_callback):
        super().__init__(
            transient_for=parent,
            heading="Transaktion bearbeiten",
            body="Ändere die Daten und speichere anschließend."
        )

        self.tx = tx
        self.on_save_callback = on_save_callback

        self.add_response("cancel", "Abbrechen")
        self.add_response("save", "Speichern")
        self.set_default_response("save")

        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)
        box.set_margin_top(12)
        box.set_margin_bottom(12)
        box.set_margin_start(12)
        box.set_margin_end(12)
        self.set_extra_child(box)

        # Datum
        lbl_date = Gtk.Label(label="Datum:")
        lbl_date.set_xalign(0)
        box.append(lbl_date)
        self.entry_date = Gtk.Entry(text=tx.date)
        box.append(self.entry_date)

        # Beschreibung
        lbl_desc = Gtk.Label(label="Beschreibung:")
        lbl_desc.set_xalign(0)
        box.append(lbl_desc)
        self.entry_desc = Gtk.Entry(text=tx.description)
        box.append(self.entry_desc)

        # Kategorie
        lbl_cat = Gtk.Label(label="Kategorie:")
        lbl_cat.set_xalign(0)
        box.append(lbl_cat)
        self.entry_category = Gtk.Entry(text=tx.category)
        box.append(self.entry_category)

        # Betrag
        lbl_amount = Gtk.Label(label="Betrag (€):")
        lbl_amount.set_xalign(0)
        box.append(lbl_amount)
        self.entry_amount = Gtk.Entry(text=f"{tx.amount}")
        box.append(self.entry_amount)

        # Typ
        lbl_type = Gtk.Label(label="Typ:")
        lbl_type.set_xalign(0)
        box.append(lbl_type)
        self.combo_type = Gtk.ComboBoxText()
        self.combo_type.append_text("Ausgabe")
        self.combo_type.append_text("Einnahme")
        self.combo_type.set_active(1 if tx.is_income else 0)
        box.append(self.combo_type)

        self.connect("response", self._on_response)

    def _on_response(self, dialog, response):
        if response != "save":
            self.destroy()
            return

        # Validierung
        try:
            amount = float(self.entry_amount.get_text().replace(",", "."))
        except ValueError:
            error = Adw.MessageDialog(
                transient_for=self.get_transient_for(),
                heading="Ungültiger Betrag",
                body="Bitte gib einen gültigen Betrag ein."
            )
            error.add_response("ok", "OK")
            error.connect("response", lambda d, r: d.destroy())
            error.show()
            return

        # Update
        self.tx.date = self.entry_date.get_text()
        self.tx.description = self.entry_desc.get_text()
        self.tx.category = self.entry_category.get_text()
        self.tx.amount = amount
        self.tx.is_income = (self.combo_type.get_active_text() == "Einnahme")

        # Callback
        self.on_save_callback(self.tx)

        self.destroy()


# ╔══════════════════════════════════════════════════════════════╗
#   DELETE DIALOG
# ╚══════════════════════════════════════════════════════════════╝

class DeleteConfirmationDialog(Adw.MessageDialog):
    def __init__(self, parent, tx: Transaction, on_delete_callback):
        super().__init__(
            transient_for=parent,
            heading="Löschen bestätigen",
            body=f"Soll die Transaktion\n\n{tx.description} ({format_euro(tx.amount)})\n\nwirklich gelöscht werden?"
        )

        self.tx = tx
        self.on_delete_callback = on_delete_callback

        self.add_response("cancel", "Abbrechen")
        self.add_response("delete", "Löschen")
        self.set_response_enabled("delete", True)
        self.set_default_response("delete")

        self.connect("response", self._on_response)

    def _on_response(self, dialog, response):
        if response == "delete":
            self.on_delete_callback(self.tx)
        self.destroy()


# ╔══════════════════════════════════════════════════════════════╗
#   TRANSAKTIONSLISTE
# ╚══════════════════════════════════════════════════════════════╝

class TransactionListWidget(Gtk.ListBox):
    """
    Eine Liste von Transaktionen, mit Kontextmenü:
    - Bearbeiten
    - Löschen
    """

    def __init__(self, parent_window, on_edit, on_delete):
        super().__init__()
        self.parent_window = parent_window
        self.on_edit = on_edit          # callback(Transaction)
        self.on_delete = on_delete      # callback(Transaction)

        self.set_selection_mode(Gtk.SelectionMode.NONE)

    # ----------------------------------------------------------
    def update_transactions(self, tx_list):
        """
        tx_list: List[Transaction]
        """
        # GTK4: ListBox hat kein get_children() mehr
        child = self.get_first_child()
        while child is not None:
            next_child = child.get_next_sibling()
            self.remove(child)
            child = next_child

        # Kopfzeile
        header = Gtk.ListBoxRow()
        hbox = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
        hbox.set_margin_top(4)
        hbox.set_margin_bottom(4)
        hbox.set_margin_start(8)
        hbox.set_margin_end(8)
        header.set_child(hbox)

        for title, expand in [
            ("Datum", False),
            ("Beschreibung", True),
            ("Kategorie", True),
            ("Betrag", False),
            ("Typ", False),
            ("", False),
        ]:
            lbl = Gtk.Label(label=title)
            lbl.add_css_class("list-header")
            lbl.set_hexpand(expand)
            lbl.set_xalign(0 if expand else 1)
            hbox.append(lbl)

        self.append(header)

        # Datenzeilen
        for tx in tx_list:
            row = Gtk.ListBoxRow()
            row.tx = tx
            box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
            box.set_margin_start(8)
            box.set_margin_end(8)
            box.set_margin_top(3)
            box.set_margin_bottom(3)
            row.set_child(box)

            # Datum
            lbl_date = Gtk.Label(label=tx.date)
            lbl_date.set_xalign(0)
            box.append(lbl_date)

            # Beschreibung
            lbl_desc = Gtk.Label(label=tx.description)
            lbl_desc.set_xalign(0)
            lbl_desc.set_hexpand(True)
            lbl_desc.set_ellipsize(3)
            box.append(lbl_desc)

            # Kategorie
            lbl_cat = Gtk.Label(label=tx.category)
            lbl_cat.set_xalign(0)
            lbl_cat.set_hexpand(True)
            lbl_cat.set_ellipsize(3)
            box.append(lbl_cat)

            # Betrag
            lbl_amount = Gtk.Label(label=format_euro(tx.amount))
            lbl_amount.set_xalign(1)
            lbl_amount.add_css_class("amount-income" if tx.is_income else "amount-expense")
            box.append(lbl_amount)

            # Typ
            type_lbl = Gtk.Label(label=("Einnahme" if tx.is_income else "Ausgabe"))
            type_lbl.set_xalign(1)
            box.append(type_lbl)

            # Menübutton
            menu_btn = Gtk.MenuButton(icon_name="open-menu-symbolic")
            menu = Gio.Menu()
            menu.append("Bearbeiten", "row.edit")
            menu.append("Löschen", "row.delete")
            menu_btn.set_menu_model(menu)

            # Actions
            group = Gio.SimpleActionGroup()
            edit_action = Gio.SimpleAction.new("edit", None)
            delete_action = Gio.SimpleAction.new("delete", None)
            group.add_action(edit_action)
            group.add_action(delete_action)
            row.insert_action_group("row", group)

            edit_action.connect("activate", lambda _a, _p, tx=tx: self.on_edit(tx))
            delete_action.connect("activate", lambda _a, _p, tx=tx: self.on_delete(tx))

            box.append(menu_btn)
            self.append(row)

# ╔══════════════════════════════════════════════════════════════╗
#   FILTER-BANNER
# ╚══════════════════════════════════════════════════════════════╝

class FilterBannerWidget(Gtk.Box):
    """
    Zeigt einen Filterstatus an, z.B.:
    "Gefiltert nach Kategorie 'Lebensmittel'  [X]"
    """

    def __init__(self):
        super().__init__(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        self.set_margin_top(8)
        self.set_margin_bottom(8)
        self.set_margin_start(8)
        self.set_margin_end(8)
        self.add_css_class("accent")

        self.label = Gtk.Label()
        self.label.set_xalign(0)
        self.label.add_css_class("ai-title")

        self.btn_clear = Gtk.Button(icon_name="window-close-symbolic")
        self.btn_clear.add_css_class("flat")

        self.append(self.label)
        self.append(self.btn_clear)

        self.set_visible(False)

    def show_category_filter(self, category: str, on_clear):
        self.label.set_text(f"Gefiltert nach Kategorie: {category}")
        self.btn_clear.connect("clicked", lambda _b: on_clear())
        self.set_visible(True)

    def show_month_filter(self, year: int, month: int, on_clear):
        self.label.set_text(f"Gefiltert nach Monat: {year}-{month:02d}")
        self.btn_clear.connect("clicked", lambda _b: on_clear())
        self.set_visible(True)

    def hide_filter(self):
        self.set_visible(False)
