#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Void Community Printer Assistant (GTK4 / optional libadwaita)
Includes:
- Service Status Check (via Process ID for non-root compatibility)
- Printer Discovery & Driverless Setup
- Manual URI Entry
- PPD Selection Support
- Printer Management (Test Page, Delete)
"""

import gi
import os
import re
import shlex
import subprocess
import threading
import configparser
import locale
from pathlib import Path

gi.require_version("Gtk", "4.0")
from gi.repository import Gtk, Gio, Gdk  # noqa: E402

# Optional: libadwaita
USE_ADW = False
try:
    gi.require_version("Adw", "1")
    from gi.repository import Adw  # type: ignore  # noqa: E402
    USE_ADW = True
except Exception:
    Adw = None  # type: ignore

# ---------------------------
# Internationalization (I18N)
# ---------------------------
TRANSLATIONS = {
    "de": {
        "Void Community Printer Assistant": "Void Community Printer-Assistent",
        "Setup printing system · Find devices · Add driverless": "Drucksystem einrichten · Geräte finden · Driverless hinzufügen",
        "Command takes too long.": "Befehl hat zu lange gedauert.",
        "Done. Press any key to close...": "Fertig. Taste drücken, um das Terminal zu schließen...",
        "Queue existing: ": "Bestehende Queue: ",
        "No terminal found. Install xfce4-terminal / kitty / konsole.": "Kein Terminal gefunden. Installiere z. B. xfce4-terminal / kitty / konsole.",
        "No terminal found.": "Kein Terminal gefunden.",
        "Terminal started.": "Terminal gestartet.",
        "Could not start terminal: ": "Konnte Terminal nicht starten: ",
        "No command specified.": "Kein Befehl angegeben.",
        "Execute action?": "Aktion ausführen?",
        "The command will be run in an external terminal.\n\nCommand:\n": "Der Befehl wird in einem externen Terminal ausgeführt.\n\nBefehl:\n",
        "Cancel": "Abbrechen",
        "Execute": "Ausführen",
        "Info": "Info",
        "Dark (click for Light)": "Dark (klicken für Light)",
        "Light (click for System)": "Light (klicken für System)",
        "System (click for Dark)": "System (klicken für Dark)",
        "Rescan": "Neu scannen",
        "Quit": "Beenden",
        "Nothing to copy.": "Nichts zum Kopieren.",
        "Copied to clipboard.": "In Zwischenablage kopiert.",
        "Step 1 · Setup Printing System": "Schritt 1 · Drucksystem einrichten",
        "Installs CUPS + filters + Avahi and enables services (runit).": "Installiert CUPS + Filter + Avahi und aktiviert die Dienste (runit).",
        "Copy command": "Befehl kopieren",
        "Run in external terminal": "Im externen Terminal ausführen",
        "Run": "Ausführen",
        "Step 2 · Find Printers & Add Driverless": "Schritt 2 · Drucker finden & driverless hinzufügen",
        "Select an IPP printer (network) and add it driverless (everywhere).": "Wähle einen IPP-Drucker (Netzwerk) aus und füge ihn treiberlos hinzu (everywhere).",
        "Add as driverless": "Als driverless hinzufügen",
        "Step 3 · Manufacturer Drivers (Optional)": "Schritt 3 · Hersteller-Treiber (optional)",
        "If driverless isn't enough: install specific driver packages. You can edit the commands.": "Falls driverless nicht reicht: installiere passende Treiberpakete. Du kannst die Befehle anpassen.",
        "Show driver packages": "Treiberpakete anzeigen",
        "Copy": "Kopieren",
        "Printer Settings": "Drucker Einstellungen",
        "Status / Log": "Status / Log",
        "Clear": "Leeren",
        "Searching for printers…": "Suche nach Druckern…",
        "No printers found": "Keine Drucker gefunden",
        "Printer turned on/connected?\nFor network printers: Avahi/CUPS active?\nClick 'Rescan'.": "Drucker eingeschaltet/verbunden?\nFür Netzwerkdrucker: Avahi/CUPS aktiv?\nKlicke auf „Neu scannen“.",
        "Source": "Quelle",
        "Detects printers & sets up driverless printing (IPP everywhere).\nAll actions run in an external terminal.\n\nDesigned by Pinguin-TV · Void-Gemeni Project": "Erkennt Drucker & richtet treiberloses Drucken (IPP everywhere) ein.\nAlle Aktionen laufen im externen Terminal.\n\nDesigned by Pinguin-TV · Void-Gemeni Projekt",
        "Note: 'pkexec' not found. You might need to add 'sudo' to commands manually.": "Hinweis: 'pkexec' nicht gefunden. Befehle ggf. mit 'sudo' im Feld ergänzen.",
        "Service Status": "Dienst-Status",
        "Running": "Läuft",
        "Stopped / Not installed": "Gestoppt / Nicht installiert",
        "Manual Connection": "Manuelle Verbindung",
        "Enter URI (e.g. ipp://192.168.1.50)": "URI eingeben (z.B. ipp://192.168.1.50)",
        "Add Manually": "Manuell hinzufügen",
        "Add with PPD...": "Mit PPD hinzufügen...",
        "Test Page": "Testseite",
        "Delete Printer": "Drucker entfernen",
        "Select PPD File": "PPD Datei auswählen",
        "PPD Files": "PPD Dateien"
    },
    "es": {
        "Void Community Printer Assistant": "Asistente de Impresora Void",
        "Setup printing system · Find devices · Add driverless": "Configurar sistema · Buscar dispositivos · Añadir sin controladores",
        "Command takes too long.": "El comando tarda demasiado.",
        "Done. Press any key to close...": "Hecho. Presione cualquier tecla para cerrar...",
        "Queue existing: ": "Cola existente: ",
        "No terminal found. Install xfce4-terminal / kitty / konsole.": "No se encontró terminal. Instale xfce4-terminal / kitty / konsole.",
        "No terminal found.": "No se encontró terminal.",
        "Terminal started.": "Terminal iniciado.",
        "Could not start terminal: ": "No se pudo iniciar el terminal: ",
        "No command specified.": "Ningún comando especificado.",
        "Execute action?": "¿Ejecutar acción?",
        "The command will be run in an external terminal.\n\nCommand:\n": "El comando se ejecutará en un terminal externo.\n\nComando:\n",
        "Cancel": "Cancelar",
        "Execute": "Ejecutar",
        "Info": "Info",
        "Dark (click for Light)": "Oscuro (clic para Claro)",
        "Light (click for System)": "Claro (clic para Sistema)",
        "System (click for Dark)": "Sistema (clic para Oscuro)",
        "Rescan": "Escanear de nuevo",
        "Quit": "Salir",
        "Nothing to copy.": "Nada que copiar.",
        "Copied to clipboard.": "Copiado al portapapeles.",
        "Step 1 · Setup Printing System": "Paso 1 · Configurar sistema de impresión",
        "Installs CUPS + filters + Avahi and enables services (runit).": "Instala CUPS + filtros + Avahi y activa los servicios (runit).",
        "Copy command": "Copiar comando",
        "Run in external terminal": "Ejecutar en terminal externo",
        "Run": "Ejecutar",
        "Step 2 · Find Printers & Add Driverless": "Paso 2 · Buscar impresoras y añadir sin controladores",
        "Select an IPP printer (network) and add it driverless (everywhere).": "Seleccione una impresora IPP (red) y añádala sin controladores (everywhere).",
        "Add as driverless": "Añadir sin controladores",
        "Step 3 · Manufacturer Drivers (Optional)": "Paso 3 · Controladores del fabricante (Opcional)",
        "If driverless isn't enough: install specific driver packages. You can edit the commands.": "Si 'driverless' no es suficiente: instale paquetes de controladores específicos. Puede editar los comandos.",
        "Show driver packages": "Mostrar paquetes de controladores",
        "Copy": "Copiar",
        "Printer Settings": "Configuración de impresora",
        "Status / Log": "Estado / Registro",
        "Clear": "Limpiar",
        "Searching for printers…": "Buscando impresoras…",
        "No printers found": "No se encontraron impresoras",
        "Printer turned on/connected?\nFor network printers: Avahi/CUPS active?\nClick 'Rescan'.": "¿Impresora encendida/conectada?\nPara impresoras de red: ¿Avahi/CUPS activo?\nHaga clic en 'Escanear de nuevo'.",
        "Source": "Fuente",
        "Detects printers & sets up driverless printing (IPP everywhere).\nAll actions run in an external terminal.\n\nDiseñado por Pinguin-TV · Proyecto Void-Gemeni": "Detecta impresoras y configura la impresión sin controladores (IPP everywhere).\nTodas las acciones se ejecutan en un terminal externo.\n\nDiseñado por Pinguin-TV · Proyecto Void-Gemeni",
        "Note: 'pkexec' not found. You might need to add 'sudo' to commands manually.": "Nota: no se encontró 'pkexec'. Es posible que deba añadir 'sudo' a los comandos manualmente.",
        "Service Status": "Estado del servicio",
        "Running": "Ejecutando",
        "Stopped / Not installed": "Detenido / No instalado",
        "Manual Connection": "Conexión Manual",
        "Enter URI (e.g. ipp://192.168.1.50)": "Introducir URI (ej. ipp://192.168.1.50)",
        "Add Manually": "Añadir manualmente",
        "Add with PPD...": "Añadir con PPD...",
        "Test Page": "Página de prueba",
        "Delete Printer": "Eliminar impresora",
        "Select PPD File": "Seleccionar archivo PPD",
        "PPD Files": "Archivos PPD"
    },
    "fr": {
        "Void Community Printer Assistant": "Assistant d'Imprimante Void",
        "Setup printing system · Find devices · Add driverless": "Configuration système · Trouver des appareils · Ajouter sans pilote",
        "Command takes too long.": "La commande prend trop de temps.",
        "Done. Press any key to close...": "Terminé. Appuyez sur une touche pour fermer...",
        "Queue existing: ": "File d'attente existante : ",
        "No terminal found. Install xfce4-terminal / kitty / konsole.": "Aucun terminal trouvé. Installez xfce4-terminal / kitty / konsole.",
        "No terminal found.": "Aucun terminal trouvé.",
        "Terminal started.": "Terminal démarré.",
        "Could not start terminal: ": "Impossible de démarrer le terminal : ",
        "No command specified.": "Aucune commande spécifiée.",
        "Execute action?": "Exécuter l'action ?",
        "The command will be run in an external terminal.\n\nCommand:\n": "La commande sera exécutée dans un terminal externe.\n\nCommande :\n",
        "Cancel": "Annuler",
        "Execute": "Exécuter",
        "Info": "Info",
        "Dark (click for Light)": "Sombre (cliquer pour Clair)",
        "Light (click for System)": "Clair (cliquer pour Système)",
        "System (click for Dark)": "Système (cliquer pour Sombre)",
        "Rescan": "Rescanner",
        "Quit": "Quitter",
        "Nothing to copy.": "Rien à copier.",
        "Copied to clipboard.": "Copié dans le presse-papiers.",
        "Step 1 · Setup Printing System": "Étape 1 · Configuration du système d'impression",
        "Installs CUPS + filters + Avahi and enables services (runit).": "Installe CUPS + filtres + Avahi et active les services (runit).",
        "Copy command": "Copier la commande",
        "Run in external terminal": "Exécuter dans un terminal externe",
        "Run": "Exécuter",
        "Step 2 · Find Printers & Add Driverless": "Étape 2 · Trouver des imprimantes & ajouter sans pilote",
        "Select an IPP printer (network) and add it driverless (everywhere).": "Sélectionnez une imprimante IPP (réseau) et ajoutez-la sans pilote (everywhere).",
        "Add as driverless": "Ajouter sans pilote",
        "Step 3 · Manufacturer Drivers (Optional)": "Étape 3 · Pilotes constructeur (Optionnel)",
        "If driverless isn't enough: install specific driver packages. You can edit the commands.": "Si 'driverless' ne suffit pas : installez des paquets de pilotes spécifiques. Vous pouvez modifier les commandes.",
        "Show driver packages": "Afficher les paquets de pilotes",
        "Copy": "Copier",
        "Printer Settings": "Paramètres d'imprimante",
        "Status / Log": "Statut / Journal",
        "Clear": "Effacer",
        "Searching for printers…": "Recherche d'imprimantes…",
        "No printers found": "Aucune imprimante trouvée",
        "Printer turned on/connected?\nFor network printers: Avahi/CUPS active?\nClick 'Rescan'.": "Imprimante allumée/connectée ?\nPour les imprimantes réseau : Avahi/CUPS actif ?\nCliquez sur 'Rescanner'.",
        "Source": "Source",
        "Detects printers & sets up driverless printing (IPP everywhere).\nAll actions run in an external terminal.\n\nDesigned by Pinguin-TV · Void-Gemeni Project": "Détecte les imprimantes et configure l'impression sans pilote (IPP everywhere).\nToutes les actions s'exécutent dans un terminal externe.\n\nConçu par Pinguin-TV · Projet Void-Gemeni",
        "Note: 'pkexec' not found. You might need to add 'sudo' to commands manually.": "Note : 'pkexec' introuvable. Vous devrez peut-être ajouter 'sudo' aux commandes manuellement.",
        "Service Status": "État du service",
        "Running": "En cours d'exécution",
        "Stopped / Not installed": "Arrêté / Non installé",
        "Manual Connection": "Connexion Manuelle",
        "Enter URI (e.g. ipp://192.168.1.50)": "Entrez l'URI (ex: ipp://192.168.1.50)",
        "Add Manually": "Ajouter manuellement",
        "Add with PPD...": "Ajouter avec PPD...",
        "Test Page": "Page de test",
        "Delete Printer": "Supprimer l'imprimante",
        "Select PPD File": "Sélectionner un fichier PPD",
        "PPD Files": "Fichiers PPD"
    }
}

def get_translation_func():
    try:
        sys_lang = locale.getdefaultlocale()[0]
        if sys_lang:
            sys_lang = sys_lang.lower()
            if sys_lang.startswith("de"):
                lang = "de"
            elif sys_lang.startswith("es"):
                lang = "es"
            elif sys_lang.startswith("fr"):
                lang = "fr"
            else:
                lang = "en"
        else:
            lang = "en"
    except Exception:
        lang = "en"

    def _(text):
        if lang in TRANSLATIONS and text in TRANSLATIONS[lang]:
            return TRANSLATIONS[lang][text]
        return text
    return _

_ = get_translation_func()

APP_ID = "org.void.printerhelper"
APP_TITLE = _("Void Community Printer Assistant")
APP_SUBTITLE = _("Setup printing system · Find devices · Add driverless")

CFG_DIR = Path.home() / ".config" / "void-printerhelper"
CFG_FILE = CFG_DIR / "settings.ini"

GENERAL_INSTALL_CMD = (
    "xbps-install -S cups cups-filters gutenprint avahi && "
    "ln -sf /etc/sv/cupsd /var/service && sv up cupsd && "
    "ln -sf /etc/sv/avahi-daemon /var/service && sv up avahi-daemon"
)

# ---------------------------
# Utilities
# ---------------------------
def which(prog: str):
    for p in os.environ.get("PATH", "").split(os.pathsep):
        f = os.path.join(p, prog)
        if os.path.isfile(f) and os.access(f, os.X_OK):
            return f
    return None


def check_process_running(process_name: str) -> bool:
    """
    Checks if a process is running using 'pgrep'.
    This works without root privileges (unlike 'sv status' on Void).
    """
    if not which("pgrep"):
        # Fallback if pgrep is missing, though unlikely on Void
        return False
    try:
        # -x matches exact name, stdout to devnull to keep it quiet
        subprocess.check_call(["pgrep", "-x", process_name], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
        return True
    except subprocess.CalledProcessError:
        return False
    except Exception:
        return False


def run_capture(cmd):
    try:
        return subprocess.check_output(cmd, stderr=subprocess.STDOUT, text=True, timeout=10)
    except subprocess.TimeoutExpired:
        return f"[Error] {(' '.join(cmd))} " + _("Command takes too long.")
    except subprocess.CalledProcessError as e:
        return e.output
    except Exception as e:
        return f"[Error] {' '.join(cmd)}: {e}"


def need_root_wrap(cmd: str) -> str:
    cmd = cmd.strip()
    needs_prefix = ("xbps-install", "ln -s", "sv ", "lpadmin", "lpoptions", "cupsctl", "systemctl", "lp ")
    if os.geteuid() == 0 or cmd.startswith(("sudo ", "pkexec ")):
        return cmd
    if cmd.startswith(needs_prefix) or (cmd.split()[:1] and cmd.split()[0] in [x.split()[0] for x in needs_prefix]):
        if which("pkexec"):
            return "pkexec " + cmd
    return cmd


def slugify(name: str) -> str:
    return re.sub(r"[^A-Za-z0-9_.-]+", "_", name).strip("_") or "Printer"


def set_clipboard_text(text: str):
    Gdk.Display.get_default().get_clipboard().set_text(text)


# ---------------------------
# Settings (Theme)
# ---------------------------
def load_theme_pref() -> str:
    try:
        if not CFG_FILE.exists():
            return "system"
        cfg = configparser.ConfigParser()
        cfg.read(CFG_FILE, encoding="utf-8")
        return cfg.get("ui", "theme", fallback="system").strip().lower()
    except Exception:
        return "system"


def save_theme_pref(value: str):
    try:
        CFG_DIR.mkdir(parents=True, exist_ok=True)
        cfg = configparser.ConfigParser()
        cfg["ui"] = {"theme": value}
        with CFG_FILE.open("w", encoding="utf-8") as f:
            cfg.write(f)
    except Exception:
        pass


# ---------------------------
# Externes Terminal
# ---------------------------
TERMINAL_CANDIDATES = [
    ("xfce4-terminal", ["xfce4-terminal", "-e"]),
    ("kitty",          ["kitty"]),
    ("konsole",        ["konsole", "-e"]),
    ("gnome-terminal", ["gnome-terminal", "--"]),
    ("alacritty",      ["alacritty", "-e"]),
    ("tilix",          ["tilix", "-e"]),
    ("qterminal",      ["qterminal", "-e"]),
    ("xterm",          ["xterm", "-e"]),
    ("foot",           ["foot", "-e"]),
    ("wezterm",        ["wezterm", "start", "--"]),
]


def find_terminal():
    for name, argv in TERMINAL_CANDIDATES:
        if which(name):
            return name, argv
    return None, None


def build_terminal_exec(cmdline: str):
    msg = _("Done. Press any key to close...")
    hold = f"echo; read -n 1 -s -r -p '{msg}' </dev/tty"
    term_name, argv = find_terminal()
    if not term_name:
        return None
    if term_name in ("kitty", "gnome-terminal", "wezterm"):
        return argv + ["bash", "-lc", cmdline + f"; {hold}"]
    full = f"bash -lc {shlex.quote(cmdline + f'; {hold}')}"
    return argv + [full]


# ---------------------------
# Drucker-Erkennung (Thread)
# ---------------------------
def find_printers():
    devices, seen = [], set()

    # lpinfo (Backends)
    if which("lpinfo"):
        out = run_capture(["bash", "-lc", "lpinfo -v"])
        for line in (l.strip() for l in out.splitlines()):
            m = re.match(r"(\S+)\s+(.+)", line)
            if not m:
                continue
            kind, rest = m.groups()
            uri = rest.strip() if "://" in rest else None
            if uri and uri not in seen:
                seen.add(uri)
                devices.append({"uri": uri, "desc": f"{kind} {uri}", "source": "lpinfo", "kind": kind})

    # lpstat (Queues)
    if which("lpstat"):
        out = run_capture(["bash", "-lc", "lpstat -a"])
        for line in (l.strip() for l in out.splitlines()):
            qname = line.split()[0] if line else ""
            if qname and qname not in seen:
                seen.add(qname)
                desc = _("Queue existing: ") + qname
                devices.append({"uri": f"queue:{qname}", "desc": desc, "source": "lpstat", "kind": "queue"})

    # avahi IPP
    if which("avahi-browse"):
        out = run_capture(["bash", "-lc", "avahi-browse -rt _ipp._tcp 2>/dev/null | grep 'address = ' -B3 -A2 || true"])
        for b in (block.strip() for block in out.split("\n--") if block.strip()):
            name_m = re.search(r"= ; (.+)", b)
            addr_m = re.search(r"address = \[(.+?)\]", b)
            port_m = re.search(r"port = \[(\d+)\]", b)
            if addr_m and port_m:
                uri = f"ipp://{addr_m.group(1)}:{port_m.group(1)}/ipp/print"
                if uri not in seen:
                    seen.add(uri)
                    name = name_m.group(1).strip() if name_m else "IPP-Printer"
                    devices.append({"uri": uri, "desc": name, "source": "avahi", "kind": "ipp"})

    return devices


def is_driverless_candidate(uri: str) -> bool:
    return uri.startswith(("ipp://", "ipps://", "dnssd://"))


def pick_icon_for_device(d: dict) -> str:
    uri = d.get("uri", "")
    src = d.get("source", "")
    kind = d.get("kind", "")

    if src == "lpstat" or uri.startswith("queue:"):
        return "document-print-symbolic"
    if kind == "ipp" or uri.startswith(("ipp://", "ipps://", "dnssd://")):
        return "printer-network-symbolic"
    if "usb" in uri:
        return "drive-removable-media-usb-symbolic"
    if "socket" in uri:
        return "network-server-symbolic"
    return "printer-symbolic"


# ---------------------------
# App
# ---------------------------
BaseApp = Adw.Application if USE_ADW else Gtk.Application


class PrinterHelper(BaseApp):
    def __init__(self):
        super().__init__(application_id=APP_ID)

        self.win = None
        self.toast_overlay = None

        # theme
        self.theme_pref = load_theme_pref()
        self.btn_theme = None

        # UI refs
        self.device_list = None
        self.device_rows = []
        self.btn_add_driverless = None
        self.btn_add_ppd = None
        self.btn_delete = None
        self.btn_test_page = None
        
        self.log_view = None
        self.lbl_svc_cups = None
        self.lbl_svc_avahi = None
        
        self.entry_manual_uri = None

        self._apply_css()

    # ---------- Styling ----------
    def _apply_css(self):
        css = """
        .card {
            border-radius: 18px;
            padding: 14px;
            background: alpha(@theme_bg_color, 0.55);
            border: 1px solid alpha(@borders, 0.35);
        }
        .section-title { font-weight: 800; font-size: 16px; letter-spacing: 0.2px; }
        .muted { opacity: 0.85; }
        .status-ok { color: #2ec27e; font-weight: bold; }
        .status-err { color: #e01b24; font-weight: bold; }
        .destructive-action { background-color: #e01b24; color: white; }
        .destructive-action:hover { background-color: #c01c28; }

        button.primary-run {
            background-image: none;
            background-color: #15b66a;
            color: #ffffff;
            border-radius: 12px;
            padding: 8px 14px;
            font-weight: 700;
        }
        button.primary-run:hover { background-color: #10975a; }
        button.primary-run:disabled { opacity: 0.45; }

        button.pill { border-radius: 999px; padding: 6px 10px; }

        textview, entry { border-radius: 12px; }

        .empty-state { padding: 18px; }
        """
        provider = Gtk.CssProvider()
        provider.load_from_data(css)
        Gtk.StyleContext.add_provider_for_display(
            Gdk.Display.get_default(),
            provider,
            Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION,
        )

    # ---------- Theme ----------
    def _update_theme_button(self):
        if not self.btn_theme:
            return
        pref = (self.theme_pref or "system").lower()
        if pref == "dark":
            self.btn_theme.set_icon_name("weather-clear-night-symbolic")
            self.btn_theme.set_tooltip_text(_("Dark (click for Light)"))
        elif pref == "light":
            self.btn_theme.set_icon_name("weather-clear-symbolic")
            self.btn_theme.set_tooltip_text(_("Light (click for System)"))
        else:
            self.btn_theme.set_icon_name("preferences-desktop-theme-symbolic")
            self.btn_theme.set_tooltip_text(_("System (click for Dark)"))

    def apply_theme(self):
        pref = (self.theme_pref or "system").lower()
        if USE_ADW:
            sm = Adw.StyleManager.get_default()
            if pref == "dark":
                sm.set_color_scheme(Adw.ColorScheme.FORCE_DARK)
            elif pref == "light":
                sm.set_color_scheme(Adw.ColorScheme.FORCE_LIGHT)
            else:
                sm.set_color_scheme(Adw.ColorScheme.DEFAULT)
        self._update_theme_button()

    def toggle_theme(self, *_):
        cur = (self.theme_pref or "system").lower()
        nxt = {"system": "dark", "dark": "light", "light": "system"}.get(cur, "system")
        self.theme_pref = nxt
        save_theme_pref(self.theme_pref)
        self.apply_theme()
        self.toast(f"Theme: {self.theme_pref}")

    # ---------- UX helpers ----------
    def _section_title(self, text: str) -> Gtk.Label:
        lbl = Gtk.Label(xalign=0.0, label=text)
        lbl.add_css_class("section-title")
        return lbl

    def toast(self, text: str):
        if USE_ADW and self.toast_overlay:
            self.toast_overlay.add_toast(Adw.Toast.new(text))
        else:
            self.append_log(f"[INFO] {text}\n")

    def append_log(self, text: str):
        buf = self.log_view.get_buffer()
        buf.insert(buf.get_end_iter(), text)
        end = buf.get_end_iter()
        mark = buf.create_mark(None, end, True)
        self.log_view.scroll_to_mark(mark, 0.0, True, 0.0, 1.0)

    def _copy_text(self, text: str):
        text = (text or "").strip()
        if not text:
            self.toast(_("Nothing to copy."))
            return
        set_clipboard_text(text)
        self.toast(_("Copied to clipboard."))

    # ---------- Service Status ----------
    def check_services_ui(self):
        # Changed to process checking for user permission compatibility
        is_cups = check_process_running("cupsd")
        is_avahi = check_process_running("avahi-daemon")
        
        def set_status(lbl, is_running):
            if is_running:
                lbl.set_label("● " + _("Running"))
                lbl.remove_css_class("status-err")
                lbl.add_css_class("status-ok")
            else:
                lbl.set_label("● " + _("Stopped / Not installed"))
                lbl.remove_css_class("status-ok")
                lbl.add_css_class("status-err")

        if self.lbl_svc_cups:
            set_status(self.lbl_svc_cups, is_cups)
        if self.lbl_svc_avahi:
            set_status(self.lbl_svc_avahi, is_avahi)

    # ---------- External run ----------
    def run_in_external_terminal(self, cmdline: str):
        term_cmd = build_terminal_exec(cmdline)
        if not term_cmd:
            self.append_log(f"\n[ERROR] {_('No terminal found. Install xfce4-terminal / kitty / konsole.')}\n")
            self.toast(_("No terminal found."))
            return
        try:
            subprocess.Popen(term_cmd)
            self.append_log(f"\n[RUN] {cmdline}\n")
            self.toast(_("Terminal started."))
        except Exception as e:
            self.append_log(f"\n[ERROR] {_('Could not start terminal: ')}{e}\n")
            self.toast(_("Terminal could not be started."))

    def confirm_and_run(self, cmd: str):
        cmd = (cmd or "").strip()
        if not cmd:
            self.toast(_("No command specified."))
            return
        cmd = need_root_wrap(cmd)

        detail = (
            _("The command will be run in an external terminal.\n\nCommand:\n") +
            f"{cmd}"
        )

        if hasattr(Gtk, "AlertDialog"):
            dlg = Gtk.AlertDialog(
                message=_("Execute action?"),
                detail=detail,
                buttons=[_("Cancel"), _("Execute")],
                default_button=1,
                cancel_button=0,
                modal=True,
            )

            def _done(dialog, res):
                try:
                    idx = dialog.choose_finish(res)
                except Exception:
                    return
                if idx == 1:
                    self.run_in_external_terminal(cmd)

            dlg.choose(self.win, None, _done)
        else:
            self.run_in_external_terminal(cmd)

    # ---------- App lifecycle ----------
    def do_activate(self):
        if self.win:
            self.win.present()
            return

        if USE_ADW:
            Adw.init()
            self.win = Adw.ApplicationWindow(application=self)
            self.win.set_title(APP_TITLE)
            self.win.set_default_size(1080, 920)
            toolbar = self._build_adw_toolbar_view()
            self.win.set_content(toolbar)
        else:
            self.win = Gtk.ApplicationWindow(application=self, title=APP_TITLE)
            self.win.set_default_size(1080, 920)
            header = self._build_gtk_headerbar()
            self.win.set_titlebar(header)
            content = self._build_content_root()
            self.win.set_child(content)

        self.apply_theme()
        self.scan_for_devices_async()
        self.check_services_ui()
        self.win.present()

    # ---------- Headerbars ----------
    def _build_gtk_headerbar(self):
        hb = Gtk.HeaderBar()

        title = Gtk.Label()
        title.set_markup(f"<span size='large'><b>{APP_TITLE}</b></span>")
        subtitle = Gtk.Label(label=APP_SUBTITLE)
        subtitle.add_css_class("muted")

        titlebox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=2)
        titlebox.append(title)
        titlebox.append(subtitle)
        hb.set_title_widget(titlebox)

        btn_about = Gtk.Button(icon_name="help-about-symbolic")
        btn_about.add_css_class("pill")
        btn_about.set_tooltip_text(_("Info"))
        btn_about.connect("clicked", lambda *_: self.on_info())

        self.btn_theme = Gtk.Button(icon_name="preferences-desktop-theme-symbolic")
        self.btn_theme.add_css_class("pill")
        self.btn_theme.connect("clicked", self.toggle_theme)
        self._update_theme_button()

        btn_rescan = Gtk.Button(icon_name="view-refresh-symbolic")
        btn_rescan.add_css_class("pill")
        btn_rescan.set_tooltip_text(_("Rescan"))
        btn_rescan.connect("clicked", lambda *_: self.scan_for_devices_async())

        btn_quit = Gtk.Button(icon_name="application-exit-symbolic")
        btn_quit.add_css_class("pill")
        btn_quit.set_tooltip_text(_("Quit"))
        btn_quit.connect("clicked", lambda *_: self.quit())

        hb.pack_start(btn_about)
        hb.pack_end(btn_quit)
        hb.pack_end(btn_rescan)
        hb.pack_end(self.btn_theme)
        return hb

    def _build_adw_toolbar_view(self):
        toolbar = Adw.ToolbarView()
        header = Adw.HeaderBar()
        title = Gtk.Label()
        title.set_markup(f"<span size='large'><b>{APP_TITLE}</b></span>")
        subtitle = Gtk.Label(label=APP_SUBTITLE)
        subtitle.add_css_class("muted")

        titlebox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=2)
        titlebox.append(title)
        titlebox.append(subtitle)
        header.set_title_widget(titlebox)

        btn_about = Gtk.Button(icon_name="help-about-symbolic")
        btn_about.add_css_class("pill")
        btn_about.set_tooltip_text(_("Info"))
        btn_about.connect("clicked", lambda *_: self.on_info())

        self.btn_theme = Gtk.Button(icon_name="preferences-desktop-theme-symbolic")
        self.btn_theme.add_css_class("pill")
        self.btn_theme.connect("clicked", self.toggle_theme)
        self._update_theme_button()

        btn_rescan = Gtk.Button(icon_name="view-refresh-symbolic")
        btn_rescan.add_css_class("pill")
        btn_rescan.set_tooltip_text(_("Rescan"))
        btn_rescan.connect("clicked", lambda *_: self.scan_for_devices_async())

        btn_quit = Gtk.Button(icon_name="application-exit-symbolic")
        btn_quit.add_css_class("pill")
        btn_quit.set_tooltip_text(_("Quit"))
        btn_quit.connect("clicked", lambda *_: self.quit())

        header.pack_start(btn_about)
        header.pack_end(btn_quit)
        header.pack_end(btn_rescan)
        header.pack_end(self.btn_theme)

        toolbar.add_top_bar(header)
        root = self._build_content_root()
        self.toast_overlay = Adw.ToastOverlay()
        self.toast_overlay.set_child(root)
        toolbar.set_content(self.toast_overlay)

        return toolbar

    # ---------- Content ----------
    def _build_content_root(self):
        scroller = Gtk.ScrolledWindow(hexpand=True, vexpand=True)
        scroller.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)

        outer = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=14)
        outer.set_margin_top(16)
        outer.set_margin_bottom(16)
        outer.set_margin_start(16)
        outer.set_margin_end(16)
        scroller.set_child(outer)

        # Step 1: Setup & Status
        card1 = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        card1.add_css_class("card")
        card1.append(self._section_title(_("Step 1 · Setup Printing System")))

        # Feature 5: Service Status
        status_box = Gtk.Box(spacing=20)
        self.lbl_svc_cups = Gtk.Label(label="CUPS")
        self.lbl_svc_avahi = Gtk.Label(label="Avahi")
        
        status_box.append(Gtk.Label(label="<b>CUPS:</b>", use_markup=True))
        status_box.append(self.lbl_svc_cups)
        status_box.append(Gtk.Label(label="<b>Avahi:</b>", use_markup=True))
        status_box.append(self.lbl_svc_avahi)
        
        card1.append(status_box)

        info1 = Gtk.Label(
            xalign=0.0,
            wrap=True,
            label=_("Installs CUPS + filters + Avahi and enables services (runit)."),
        )
        info1.add_css_class("muted")

        self.entry_general_install = Gtk.Entry()
        self.entry_general_install.set_text(GENERAL_INSTALL_CMD)

        btnrow1 = Gtk.Box(spacing=10)
        btn_copy1 = Gtk.Button(icon_name="edit-copy-symbolic")
        btn_copy1.add_css_class("pill")
        btn_copy1.set_tooltip_text(_("Copy command"))
        btn_copy1.connect("clicked", lambda *_: self._copy_text(self.entry_general_install.get_text()))
        btn_run1 = Gtk.Button(label=_("Run"))
        btn_run1.add_css_class("primary-run")
        btn_run1.set_tooltip_text(_("Run in external terminal"))
        btn_run1.connect("clicked", lambda *_: self.confirm_and_run(self.entry_general_install.get_text()))

        btnrow1.append(btn_copy1)
        btnrow1.append(btn_run1)
        card1.append(info1)
        card1.append(self.entry_general_install)
        card1.append(btnrow1)

        # Step 2: Devices & Manual
        card2 = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        card2.add_css_class("card")
        card2.append(self._section_title(_("Step 2 · Find Printers & Add Driverless")))

        info2 = Gtk.Label(
            xalign=0.0,
            wrap=True,
            label=_("Select an IPP printer (network) and add it driverless (everywhere)."),
        )
        info2.add_css_class("muted")

        self.device_list = Gtk.ListBox()
        self.device_list.set_selection_mode(Gtk.SelectionMode.SINGLE)
        self.device_list.connect("row-selected", self.on_device_selected)

        # Action Buttons Area
        btnrow2 = Gtk.Box(spacing=10)
        btnrow2.set_margin_top(8)

        btn_rescan_inline = Gtk.Button(label=_("Rescan"), icon_name="view-refresh-symbolic")
        btn_rescan_inline.add_css_class("pill")
        btn_rescan_inline.connect("clicked", lambda *_: self.scan_for_devices_async())
        
        # New: Multiple Action Buttons
        self.btn_add_driverless = Gtk.Button(label=_("Add as driverless"))
        self.btn_add_driverless.add_css_class("primary-run")
        self.btn_add_driverless.set_sensitive(False)
        self.btn_add_driverless.connect("clicked", lambda *_: self.on_add_driverless())

        # Feature 4: PPD Selection
        self.btn_add_ppd = Gtk.Button(label=_("Add with PPD..."))
        self.btn_add_ppd.set_sensitive(False)
        self.btn_add_ppd.connect("clicked", self.on_add_ppd)

        # Feature 1: Test Page
        self.btn_test_page = Gtk.Button(label=_("Test Page"))
        self.btn_test_page.set_sensitive(False)
        self.btn_test_page.connect("clicked", self.on_test_page)

        # Feature 2: Delete Printer
        self.btn_delete = Gtk.Button(label=_("Delete Printer"))
        self.btn_delete.add_css_class("destructive-action")
        self.btn_delete.set_sensitive(False)
        self.btn_delete.connect("clicked", self.on_delete_printer)

        btnrow2.append(btn_rescan_inline)
        btnrow2.append(self.btn_add_driverless)
        btnrow2.append(self.btn_add_ppd)
        btnrow2.append(self.btn_test_page)
        btnrow2.append(self.btn_delete)

        # Feature 3: Manual Add
        manual_exp = Gtk.Expander(label=_("Manual Connection"))
        manual_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)
        manual_box.set_margin_top(8)
        
        self.entry_manual_uri = Gtk.Entry()
        self.entry_manual_uri.set_placeholder_text(_("Enter URI (e.g. ipp://192.168.1.50)"))
        
        btn_manual_add = Gtk.Button(label=_("Add Manually"))
        btn_manual_add.connect("clicked", self.on_manual_add)
        
        manual_box.append(self.entry_manual_uri)
        manual_box.append(btn_manual_add)
        manual_exp.set_child(manual_box)

        card2.append(info2)
        card2.append(self.device_list)
        card2.append(btnrow2)
        card2.append(manual_exp)

        # Step 3: Optional driver packages
        card3 = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        card3.add_css_class("card")
        card3.append(self._section_title(_("Step 3 · Manufacturer Drivers (Optional)")))

        hint3 = Gtk.Label(
            xalign=0.0,
            wrap=True,
            label=_("If driverless isn't enough: install specific driver packages. You can edit the commands."),
        )
        hint3.add_css_class("muted")
        card3.append(hint3)

        exp = Gtk.Expander(label=_("Show driver packages"))
        exp.set_expanded(True)

        grid = Gtk.Grid(column_spacing=12, row_spacing=12)
        grid.set_margin_top(10)

        def driver_card(title: str, default_cmd: str):
            box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)
            box.add_css_class("card")
            t = Gtk.Label(xalign=0.0)
            t.set_markup(f"<b>{title}</b>")
            entry = Gtk.Entry()
            entry.set_text(default_cmd)
            btnrow = Gtk.Box(spacing=10)
            bcopy = Gtk.Button(icon_name="edit-copy-symbolic")
            bcopy.add_css_class("pill")
            bcopy.set_tooltip_text(_("Copy"))
            bcopy.connect("clicked", lambda *_: self._copy_text(entry.get_text()))
            brun = Gtk.Button(label=_("Run"))
            brun.add_css_class("primary-run")
            brun.set_tooltip_text(_("Run in external terminal"))
            brun.connect("clicked", lambda *_: self.confirm_and_run(entry.get_text()))
            btnrow.append(bcopy)
            btnrow.append(brun)
            box.append(t)
            box.append(entry)
            box.append(btnrow)
            return box, entry

        hp_box, self.entry_driver_hp = driver_card("HP", "xbps-install -S hplip hplip-gui")
        epson_box, self.entry_driver_epson = driver_card("Epson", "xbps-install -S epson-inkjet-printer-escpr gutenprint")
        bro_box, self.entry_driver_brother = driver_card("Brother", "xbps-install -S foomatic-db foomatic-db-nonfree brother-brlaser")
        canon_box, self.entry_driver_canon = driver_card("Canon", "xbps-install -S cnijfilter2 gutenprint foomatic-db")
        cfg_box, self.entry_syscfg = driver_card(_("Printer Settings"), "system-config-printer")

        grid.attach(hp_box, 0, 0, 1, 1)
        grid.attach(epson_box, 1, 0, 1, 1)
        grid.attach(bro_box, 2, 0, 1, 1)
        grid.attach(canon_box, 0, 1, 1, 1)
        grid.attach(cfg_box, 1, 1, 1, 1)

        exp.set_child(grid)
        card3.append(exp)

        # Log
        card4 = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        card4.add_css_class("card")
        card4.append(self._section_title(_("Status / Log")))

        self.log_view = Gtk.TextView(editable=False, monospace=True, wrap_mode=Gtk.WrapMode.WORD_CHAR)
        self.log_view.set_vexpand(True)

        log_scroll = Gtk.ScrolledWindow(hexpand=True, vexpand=True)
        log_scroll.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
        log_scroll.set_child(self.log_view)
        log_scroll.set_min_content_height(200)

        log_btns = Gtk.Box(spacing=10)
        btn_clear = Gtk.Button(label=_("Clear"), icon_name="edit-clear-all-symbolic")
        btn_clear.add_css_class("pill")
        btn_clear.connect("clicked", lambda *_: self.log_view.get_buffer().set_text(""))
        btn_copy_log = Gtk.Button(label=_("Copy"), icon_name="edit-copy-symbolic")
        btn_copy_log.add_css_class("pill")
        btn_copy_log.connect(
            "clicked",
            lambda *_: self._copy_text(self.log_view.get_buffer().get_text(*self.log_view.get_buffer().get_bounds(), True)),
        )

        log_btns.append(btn_clear)
        log_btns.append(btn_copy_log)
        card4.append(log_scroll)
        card4.append(log_btns)

        outer.append(card1)
        outer.append(card2)
        outer.append(card3)
        outer.append(card4)
        return scroller

    # ---------------------------
    # Scan / Device list UI
    # ---------------------------
    def _device_list_clear(self):
        while child := self.device_list.get_first_child():
            self.device_list.remove(child)
        self.device_rows.clear()
        self.btn_add_driverless.set_sensitive(False)
        self.btn_add_ppd.set_sensitive(False)
        self.btn_delete.set_sensitive(False)
        self.btn_test_page.set_sensitive(False)

    def scan_for_devices_async(self):
        if not self.device_list:
            return
        
        # Check services again on rescan
        self.check_services_ui()

        self._device_list_clear()

        # loading row
        loading = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
        loading.set_margin_top(10)
        loading.set_margin_bottom(10)

        sp = Gtk.Spinner(spinning=True)
        sp.set_size_request(22, 22)
        lbl = Gtk.Label(label=_("Searching for printers…"), xalign=0.0)
        lbl.add_css_class("muted")

        loading.append(sp)
        loading.append(lbl)

        row = Gtk.ListBoxRow()
        row.set_child(loading)
        self.device_list.append(row)

        def worker():
            devices = find_printers()
            Gio.Application.get_default().get_active_window()
            Gtk.Widget
            Gtk.Application.get_default()
            from gi.repository import GLib
            GLib.idle_add(self._update_device_list_ui, devices)

        threading.Thread(target=worker, daemon=True).start()

    def _update_device_list_ui(self, devices):
        self._device_list_clear()

        if not devices:
            empty = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)
            empty.add_css_class("empty-state")

            icon = Gtk.Image.new_from_icon_name("edit-find-replace-symbolic")
            icon.set_pixel_size(56)

            t = Gtk.Label()
            t.set_markup(f"<b>{_('No printers found')}</b>")

            s = Gtk.Label(
                label=_("Printer turned on/connected?\nFor network printers: Avahi/CUPS active?\nClick 'Rescan'."),
                wrap=True,
                justify=Gtk.Justification.CENTER,
            )
            s.add_css_class("muted")

            empty.append(icon)
            empty.append(t)
            empty.append(s)

            row = Gtk.ListBoxRow()
            row.set_child(empty)
            self.device_list.append(row)
            self.append_log("[SCAN] " + _("No printers found") + "\n")
            return

        for d in devices:
            uri = d["uri"]
            desc = d["desc"]
            src = d["source"]

            icon_name = pick_icon_for_device(d)
            icon = Gtk.Image.new_from_icon_name(icon_name)
            icon.set_pixel_size(20)

            vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=2)
            title = Gtk.Label(xalign=0.0)
            title.set_markup(f"<b>{uri}</b>")

            sub = Gtk.Label(xalign=0.0, wrap=True)
            src_label = _("Source")
            sub.set_text(f"{desc}  ·  {src_label}: {src}")
            sub.add_css_class("muted")

            vbox.append(title)
            vbox.append(sub)

            h = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
            h.set_margin_top(8)
            h.set_margin_bottom(8)
            h.append(icon)
            h.append(vbox)

            row = Gtk.ListBoxRow()
            row.set_child(h)
            self.device_list.append(row)
            self.device_rows.append((row, uri, desc))

        self.append_log(f"[SCAN] {_('Searching for printers…')} -> {len(devices)}\n")
        return

    # ---------------------------
    # Callbacks & Logic
    # ---------------------------
    def on_device_selected(self, _listbox, *_args):
        sel = self.device_list.get_selected_row() if self.device_list else None
        if not sel or not self.device_rows:
            self.btn_add_driverless.set_sensitive(False)
            self.btn_add_ppd.set_sensitive(False)
            self.btn_delete.set_sensitive(False)
            self.btn_test_page.set_sensitive(False)
            return

        uri = next((u for r, u, _d in self.device_rows if r is sel), "")
        
        is_queue = uri.startswith("queue:")
        
        # New Devices (Driverless/PPD)
        self.btn_add_driverless.set_sensitive(bool(uri and is_driverless_candidate(uri) and not is_queue))
        self.btn_add_ppd.set_sensitive(bool(uri and not is_queue))
        
        # Existing Queues (Delete/Test)
        self.btn_delete.set_sensitive(is_queue)
        self.btn_test_page.set_sensitive(is_queue)
        
        # UI Visiblity Toggle (Optional but cleaner)
        self.btn_add_driverless.set_visible(not is_queue)
        self.btn_add_ppd.set_visible(not is_queue)
        self.btn_delete.set_visible(is_queue)
        self.btn_test_page.set_visible(is_queue)

    def _get_selected_uri_and_name(self):
        sel = self.device_list.get_selected_row() if self.device_list else None
        if not sel:
            return None, None
        uri = next((u for r, u, _d in self.device_rows if r is sel), None)
        if not uri:
            return None, None
        
        if uri.startswith("queue:"):
            name = uri.replace("queue:", "")
            return uri, name
        else:
            name = slugify(re.sub(r"^(dnssd://|ipps?://)", "", uri).split("/")[0])
            return uri, name

    def on_add_driverless(self):
        uri, qname = self._get_selected_uri_and_name()
        if not uri:
            return
        cmd = f"lpadmin -p {qname} -E -v '{uri}' -m everywhere && lpoptions -d {qname}"
        self.confirm_and_run(cmd)

    def on_manual_add(self, *_):
        raw_uri = self.entry_manual_uri.get_text().strip()
        if not raw_uri:
            self.toast("Empty URI")
            return
        # Basic Slugify
        qname = "Manual_Printer_" + slugify(raw_uri.replace("://", "_"))
        # Using driverless everywhere for manual IPP
        cmd = f"lpadmin -p {qname} -E -v '{raw_uri}' -m everywhere"
        self.confirm_and_run(cmd)

    def on_test_page(self, *_):
        uri, qname = self._get_selected_uri_and_name()
        if not qname:
            return
        # Try generic test page command
        cmd = f"lp -d {qname} /usr/share/cups/data/testprint"
        self.confirm_and_run(cmd)

    def on_delete_printer(self, *_):
        uri, qname = self._get_selected_uri_and_name()
        if not qname:
            return
        cmd = f"lpadmin -x {qname}"
        self.confirm_and_run(cmd)

    def on_add_ppd(self, *_):
        uri, qname = self._get_selected_uri_and_name()
        if not uri:
            return

        def on_file_chosen(dialog, response):
            if response == Gtk.ResponseType.ACCEPT:
                f = dialog.get_file()
                ppd_path = f.get_path()
                cmd = f"lpadmin -p {qname} -E -v '{uri}' -P '{ppd_path}'"
                self.confirm_and_run(cmd)
        
        fcd = Gtk.FileChooserNative(
            title=_("Select PPD File"),
            transient_for=self.win,
            action=Gtk.FileChooserAction.OPEN
        )
        filter_ppd = Gtk.FileFilter()
        filter_ppd.set_name(_("PPD Files"))
        filter_ppd.add_pattern("*.ppd")
        fcd.add_filter(filter_ppd)
        
        fcd.connect("response", on_file_chosen)
        fcd.show()

    def on_info(self):
        detail = _("Detects printers & sets up driverless printing (IPP everywhere).\nAll actions run in an external terminal.\n\nDesigned by Pinguin-TV · Void-Gemeni Project")
        if hasattr(Gtk, "AlertDialog"):
            dlg = Gtk.AlertDialog(message=APP_TITLE, detail=detail, buttons=["OK"], modal=True)
            dlg.show(self.win)
        else:
            self.append_log("[INFO] " + detail.replace("\n", " ") + "\n")


def main():
    if not which("pkexec"):
        print(_("Note: 'pkexec' not found. You might need to add 'sudo' to commands manually."))
    app = PrinterHelper()
    app.run(None)


if __name__ == "__main__":
    main()
