#!/usr/bin/env python3
# scanner_manager_unrestricted.py
# Eine radikal vereinfachte Version, bei der jeder Button immer klickbar ist.

import gi
gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
from gi.repository import Gtk, Adw, GLib
import sys
import subprocess
import threading
import getpass

# =============================================================================
#  GTK4 / Libadwaita GUI-Anwendung
# =============================================================================
class ScannerManagerApp(Adw.Application):
    # Paketlisten
    SANE_PACKAGES = ['sane-backends']
    SIMPLE_SCAN_PACKAGES = ['simple-scan']
    XSANE_PACKAGES = ['xsane']
    SKANLITE_PACKAGES = ['skanlite']
    GSCAN2PDF_PACKAGES = ['gscan2pdf']
    ISCAN_PACKAGES = ['iscan', 'iscan-data']
    IMAGESCAN_PACKAGES = ['imagescan']

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.connect('activate', self.on_activate)

    def on_activate(self, app):
        self.win = Gtk.ApplicationWindow(application=app, title="Void Scanner Manager")
        self.win.set_default_size(650, 1000)
        
        header = Adw.HeaderBar()
        self.win.set_titlebar(header)
        dark_mode_switch = Gtk.Switch(valign=Gtk.Align.CENTER)
        dark_mode_switch.connect("notify::active", self.on_dark_mode_toggled)
        header.pack_end(dark_mode_switch)
        
        self.button_size_group = Gtk.SizeGroup.new(Gtk.SizeGroupMode.HORIZONTAL)
        page = Adw.PreferencesPage()
        self.win.set_child(page)

        # --- JEDE SEKTION IST IMMER AKTIV ---

        # GRUPPE 1: SANE-DIENSTE
        self._create_section(page, title="System-Dienste (SANE)", description="Installiert das SANE-Basissystem.", packages=self.SANE_PACKAGES, name="SANE")

        # GRUPPE 2: BERECHTIGUNGEN
        permissions_group = Adw.PreferencesGroup(title="Benutzerberechtigungen", description="Fügt Ihren Benutzer zur 'scanner'-Gruppe hinzu.")
        page.add(permissions_group)
        permissions_row = Adw.ActionRow()
        permissions_group.add(permissions_row)
        permissions_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        permissions_spinner = Gtk.Spinner()
        permissions_button = Gtk.Button(label="Berechtigung erteilen")
        permissions_button.connect('clicked', self.on_grant_permissions_clicked, permissions_spinner)
        self.button_size_group.add_widget(permissions_button)
        permissions_box.append(permissions_spinner)
        permissions_box.append(permissions_button)
        permissions_row.add_suffix(permissions_box)
        
        # GRUPPE 3: SCAN-PROGRAMME
        page.add(Adw.PreferencesGroup(title="Allgemeine Scan-Programme"))
        self._create_section(page, title="Simple Scan", packages=self.SIMPLE_SCAN_PACKAGES, name="Simple Scan")
        self._create_section(page, title="XSane (für Fortgeschrittene)", packages=self.XSANE_PACKAGES, name="XSane")
        self._create_section(page, title="Skanlite (für KDE)", packages=self.SKANLITE_PACKAGES, name="Skanlite")
        self._create_section(page, title="gscan2pdf (mehrseitige PDFs)", packages=self.GSCAN2PDF_PACKAGES, name="gscan2pdf")
        
        # GRUPPE 4: HERSTELLERTREIBER
        page.add(Adw.PreferencesGroup(title="Spezifische Herstellertreiber"))
        self._create_section(page, title="Epson Treiber (iscan)", description="Legacy-Treiber für viele ältere Epson-Modelle.", packages=self.ISCAN_PACKAGES, name="Epson iscan")
        self._create_section(page, title="Epson Tool (imagescan)", description="Moderne Anwendung für neuere Epson-Scanner.", packages=self.IMAGESCAN_PACKAGES, name="Epson imagescan")

        self.win.present()

    def _create_section(self, parent_page, title, description=None, packages=None, name=None):
        group = Adw.PreferencesGroup(title=title, description=description)
        parent_page.add(group)
        row = Adw.ActionRow()
        group.add(row)
        box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        spinner = Gtk.Spinner()
        install_button = Gtk.Button(label="Installieren")
        uninstall_button = Gtk.Button(label="Deinstallieren", css_classes=["destructive-action"])
        
        self.button_size_group.add_widget(install_button)
        self.button_size_group.add_widget(uninstall_button)
        
        install_button.connect('clicked', lambda b: self.run_package_operation(packages, name, b, spinner, is_install=True))
        uninstall_button.connect('clicked', lambda b: self.run_package_operation(packages, name, b, spinner, is_install=False))
        
        box.append(spinner)
        box.append(install_button)
        box.append(uninstall_button)
        row.add_suffix(box)

    def on_dark_mode_toggled(self, switch, _): Adw.StyleManager.get_default().set_color_scheme(Adw.ColorScheme.FORCE_DARK if switch.get_active() else Adw.ColorScheme.FORCE_LIGHT)
        
    def on_grant_permissions_clicked(self, button, spinner):
        username = getpass.getuser()
        command = ["pkexec", "usermod", "-aG", "scanner", username]
        spinner.start()
        button.set_sensitive(False)
        thread = threading.Thread(target=self._worker_thread, args=(command, "Benutzerberechtigung", spinner, button))
        thread.daemon = True
        thread.start()

    def run_package_operation(self, package_list, name, button, spinner, is_install):
        action = "install" if is_install else "remove"
        command_str = ' '.join(package_list)
        command = ["pkexec", f"xbps-{action}", "-S" if is_install else "-R", command_str]
        
        spinner.start()
        button.get_parent().set_sensitive(False)
        thread = threading.Thread(target=self._worker_thread, args=(command, name, spinner, button))
        thread.daemon = True
        thread.start()
        
    def _worker_thread(self, command, name, spinner, button):
        try:
            # xbps-install kann mehrere Pakete als getrennte Argumente annehmen.
            # Wir splitten den letzten Teil des Befehls, falls es mehrere Pakete sind.
            if 'install' in command[1] and ' ' in command[-1]:
                 final_command = command[:-1] + command[-1].split()
            else:
                 final_command = command
            subprocess.run(final_command, capture_output=True, text=True, check=True)
            GLib.idle_add(self._operation_success, name, spinner, button)
        except (subprocess.CalledProcessError, FileNotFoundError) as e:
            error_message = "Fehler: 'pkexec' nicht gefunden." if isinstance(e, FileNotFoundError) else e.stderr
            GLib.idle_add(self._operation_failure, error_message, spinner, button)
            
    def _operation_success(self, name, spinner, button):
        spinner.stop()
        button.get_parent().set_sensitive(True)
        secondary_text = ""
        if name == "Benutzerberechtigung":
            secondary_text = "WICHTIG: Sie müssen sich jetzt abmelden und wieder anmelden, damit die Änderung wirksam wird."
        dialog = Gtk.MessageDialog(transient_for=self.win, modal=True, message_type=Gtk.MessageType.INFO,
                                   buttons=Gtk.ButtonsType.OK, text=f"Aktion '{name}' erfolgreich ausgeführt!",
                                   secondary_text=secondary_text)
        dialog.connect("response", lambda d, r: d.destroy())
        dialog.present()
        
    def _operation_failure(self, error_message, spinner, button):
        spinner.stop()
        button.get_parent().set_sensitive(True)
        dialog = Gtk.MessageDialog(transient_for=self.win, modal=True, message_type=Gtk.MessageType.ERROR,
                                   buttons=Gtk.ButtonsType.OK, text="Fehler bei der Operation",
                                   secondary_text=error_message)
        dialog.connect("response", lambda d, r: d.destroy())
        dialog.present()

if __name__ == "__main__":
    app = ScannerManagerApp(application_id="dev.void.ScannerManager")
    sys.exit(app.run(sys.argv))
