#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Void Community Scanner Assistant (GTK4 / optional libadwaita)
- Multi-language support (EN, DE, ES, FR)
- GNOME/Adwaita layout
- External terminal execution
- User permission management ('scanner' group) + Logout Warning
- App installation helper
- Specific support for Brother/AirScan, Epson and ipp-usb
- Auto-enable services (ipp-usb) upon installation
- Diagnostics: scanimage -L, lsusb, Service Status
"""

import gi
import os
import shlex
import subprocess
import getpass
import configparser
import threading
from pathlib import Path

gi.require_version("Gtk", "4.0")
from gi.repository import Gtk, Gdk, Gio, GLib  # noqa: E402

# Optional: libadwaita
USE_ADW = False
try:
    gi.require_version("Adw", "1")
    from gi.repository import Adw  # type: ignore  # noqa: E402
    USE_ADW = True
except Exception:
    Adw = None  # type: ignore

APP_ID = "org.void.scanner_helper"

CFG_DIR = Path.home() / ".config" / "void-scannerhelper"
CFG_FILE = CFG_DIR / "settings.ini"

SCANNER_PACKAGES = {
    "SANE": ["sane", "libsane"],
    "Simple Scan": ["simple-scan"],
    "XSane": ["xsane"],
    "Skanlite": ["skanlite"],
    "gscan2pdf": ["gscan2pdf"],
    "Epson IScan": ["iscan", "iscan-data"],
    "Epson ImageScan": ["imagescan"],
    "Epson (Epkowa)": ["sane-epkowa"],
    "Brother (AirScan)": ["sane-airscan"],
    "IPP-over-USB": ["ipp-usb"], 
}

# ---------------------------
# Internationalization (I18N)
# ---------------------------
TRANSLATIONS = {
    "de": {
        "Void Scanner Assistant": "Void Scanner Assistent",
        "SANE · Apps · Permissions · Drivers": "SANE · Apps · Berechtigungen · Treiber",
        "Command takes too long.": "Befehl hat zu lange gedauert.",
        "Done. Press any key to close...": "Fertig. Taste drücken, um das Terminal zu schließen...",
        "No terminal found. Install xfce4-terminal / kitty / konsole.": "Kein Terminal gefunden. Installiere z. B. xfce4-terminal / kitty / konsole.",
        "No terminal found.": "Kein Terminal gefunden.",
        "Terminal started.": "Terminal gestartet.",
        "Could not start terminal: ": "Konnte Terminal nicht starten: ",
        "No command specified.": "Kein Befehl angegeben.",
        "Execute action?": "Aktion ausführen?",
        "The command will be run in an external terminal.\n\nCommand:\n": "Der Befehl wird in einem externen Terminal ausgeführt.\n\nBefehl:\n",
        "Cancel": "Abbrechen",
        "Execute": "Ausführen",
        "Info": "Info",
        "Dark (click for Light)": "Dark (klicken für Light)",
        "Light (click for System)": "Light (klicken für System)",
        "System (click for Dark)": "System (klicken für Dark)",
        "Quit": "Beenden",
        "Nothing to copy.": "Nichts zum Kopieren.",
        "Copied to clipboard.": "In Zwischenablage kopiert.",
        "Step 1 · Scanner Basics (SANE)": "Schritt 1 · Scanner-Grundlagen (SANE)",
        "Installs SANE backends – the foundation for scanner support.": "Installiert 'sane' und 'libsane' – die Basis für Scanner-Unterstützung.",
        "Step 2 · User Permissions": "Schritt 2 · Benutzerberechtigungen",
        "Adds your user to the 'scanner' group so devices can be used without root.": "Fügt deinen Benutzer zur Gruppe 'scanner' hinzu, damit Geräte ohne Root nutzbar sind.",
        "Grant Permissions": "Berechtigung erteilen",
        "Step 3 · Install Scanning Apps": "Schritt 3 · Scan-Apps installieren",
        "Choose an app. Simple Scan is usually the best default choice.": "Wähle eine App. Simple Scan ist meist die beste Standardwahl.",
        "Simple & fast.": "Einfach & schnell.",
        "Advanced / classic.": "Fortgeschritten / klassisch.",
        "KDE tool.": "KDE-Tool.",
        "Multipage PDFs.": "Mehrseitige PDFs.",
        "Step 4 · Manufacturer Drivers (Optional)": "Schritt 4 · Herstellertreiber (optional)",
        "Only needed if your device isn't detected properly by SANE.": "Nur nötig, wenn dein Gerät mit SANE nicht sauber erkannt wird.",
        "Show Manufacturer Drivers": "Hersteller-Treiber anzeigen",
        "Legacy for older models.": "Legacy für ältere Modelle.",
        "Newer models.": "Neuere Modelle.",
        "Status / Log": "Status / Log",
        "Clear": "Leeren",
        "Copy": "Kopieren",
        "Install": "Installieren",
        "Uninstall": "Deinstallieren",
        "No packages specified.": "Keine Pakete angegeben.",
        "Scanner Assistant ready.": "Scanner Assistent bereit.",
        "Designed by Pinguin-TV · Void-Gemeni Project": "Designed by Pinguin-TV · Void-Gemeni Projekt",
        "Note: 'pkexec' not found. You might need to add 'sudo' to commands manually.": "Hinweis: 'pkexec' nicht gefunden. Befehle ggf. mit 'sudo' im Feld ergänzen.",
        "Brother / AirScan": "Brother / AirScan",
        "Essential for modern network scanners (eSCL/WSD).": "Essentiell für moderne Netzwerk-Scanner (eSCL/WSD).",
        "Epson (Epkowa)": "Epson (Epkowa)",
        "External backend (often needed for specific Epson models).": "Externes Backend (oft für spezifische Epson-Modelle nötig).",
        "IPP-over-USB": "IPP-over-USB",
        "Allows driverless scanning via USB (modern HP/Brother/Canon).": "Erlaubt treiberloses Scannen via USB (moderne HP/Brother/Canon).",
        "Scan for Devices": "Scanner suchen",
        "List USB": "USB listen",
        "Service Status": "Dienst-Status",
        "Running": "Läuft",
        "Stopped / Not installed": "Gestoppt / Nicht installiert",
        "Important: Re-login required!": "Wichtig: Abmelden erforderlich!",
        "You have been added to the 'scanner' group.\nPlease logout and login again for this to take effect.": "Du wurdest zur Gruppe 'scanner' hinzugefügt.\nBitte melde dich ab und wieder an, damit die Änderung wirksam wird.",
        "Searching...": "Suche läuft...",
    },
    "es": {
        "Void Scanner Assistant": "Asistente de Escáner Void",
        "SANE · Apps · Permissions · Drivers": "SANE · Apps · Permisos · Controladores",
        "Command takes too long.": "El comando tarda demasiado.",
        "Done. Press any key to close...": "Hecho. Presione cualquier tecla para cerrar...",
        "No terminal found. Install xfce4-terminal / kitty / konsole.": "No se encontró terminal. Instale xfce4-terminal / kitty / konsole.",
        "No terminal found.": "No se encontró terminal.",
        "Terminal started.": "Terminal iniciado.",
        "Could not start terminal: ": "No se pudo iniciar el terminal: ",
        "No command specified.": "Ningún comando especificado.",
        "Execute action?": "¿Ejecutar acción?",
        "The command will be run in an external terminal.\n\nCommand:\n": "El comando se ejecutará en un terminal externo.\n\nComando:\n",
        "Cancel": "Cancelar",
        "Execute": "Ejecutar",
        "Info": "Info",
        "Dark (click for Light)": "Oscuro (clic para Claro)",
        "Light (click for System)": "Claro (clic para Sistema)",
        "System (click for Dark)": "Sistema (clic para Oscuro)",
        "Quit": "Salir",
        "Nothing to copy.": "Nada que copiar.",
        "Copied to clipboard.": "Copiado al portapapeles.",
        "Step 1 · Scanner Basics (SANE)": "Paso 1 · Conceptos básicos (SANE)",
        "Installs SANE backends – the foundation for scanner support.": "Instala 'sane' y 'libsane': la base para el soporte de escáneres.",
        "Step 2 · User Permissions": "Paso 2 · Permisos de usuario",
        "Adds your user to the 'scanner' group so devices can be used without root.": "Añade su usuario al grupo 'scanner' para usar dispositivos sin root.",
        "Grant Permissions": "Conceder permisos",
        "Step 3 · Install Scanning Apps": "Paso 3 · Instalar apps de escaneo",
        "Choose an app. Simple Scan is usually the best default choice.": "Elija una aplicación. Simple Scan suele ser la mejor opción predeterminada.",
        "Simple & fast.": "Simple y rápido.",
        "Advanced / classic.": "Avanzado / clásico.",
        "KDE tool.": "Herramienta KDE.",
        "Multipage PDFs.": "PDFs multipágina.",
        "Step 4 · Manufacturer Drivers (Optional)": "Paso 4 · Controladores del fabricante (Opcional)",
        "Only needed if your device isn't detected properly by SANE.": "Solo necesario si SANE no detecta correctamente su dispositivo.",
        "Show Manufacturer Drivers": "Mostrar controladores del fabricante",
        "Legacy for older models.": "Legacy para modelos antiguos.",
        "Newer models.": "Modelos más nuevos.",
        "Status / Log": "Estado / Registro",
        "Clear": "Limpiar",
        "Copy": "Copiar",
        "Install": "Instalar",
        "Uninstall": "Desinstalar",
        "No packages specified.": "No se especificaron paquetes.",
        "Scanner Assistant ready.": "Asistente de escáner listo.",
        "Designed by Pinguin-TV · Void-Gemeni Project": "Diseñado por Pinguin-TV · Proyecto Void-Gemeni",
        "Note: 'pkexec' not found. You might need to add 'sudo' to commands manually.": "Nota: 'pkexec' no encontrado. Puede que necesite añadir 'sudo' manualmente.",
        "Brother / AirScan": "Brother / AirScan",
        "Essential for modern network scanners (eSCL/WSD).": "Esencial para escáneres de red modernos (eSCL/WSD).",
        "Epson (Epkowa)": "Epson (Epkowa)",
        "External backend (often needed for specific Epson models).": "Backend externo (a menudo necesario para modelos Epson específicos).",
        "IPP-over-USB": "IPP-over-USB",
        "Allows driverless scanning via USB (modern HP/Brother/Canon).": "Permite escanear sin drivers por USB (HP/Brother/Canon modernos).",
        "Scan for Devices": "Buscar dispositivos",
        "List USB": "Listar USB",
        "Service Status": "Estado del servicio",
        "Running": "Ejecutando",
        "Stopped / Not installed": "Detenido / No instalado",
        "Important: Re-login required!": "¡Importante: Re-inicio de sesión necesario!",
        "You have been added to the 'scanner' group.\nPlease logout and login again for this to take effect.": "Se le ha añadido al grupo 'scanner'.\nPor favor cierre sesión e inicie de nuevo para aplicar los cambios.",
        "Searching...": "Buscando...",
    },
    "fr": {
        "Void Scanner Assistant": "Assistant Scanner Void",
        "SANE · Apps · Permissions · Drivers": "SANE · Apps · Permissions · Pilotes",
        "Command takes too long.": "La commande prend trop de temps.",
        "Done. Press any key to close...": "Terminé. Appuyez sur une touche pour fermer...",
        "No terminal found. Install xfce4-terminal / kitty / konsole.": "Aucun terminal trouvé. Installez xfce4-terminal / kitty / konsole.",
        "No terminal found.": "Aucun terminal trouvé.",
        "Terminal started.": "Terminal démarré.",
        "Could not start terminal: ": "Impossible de démarrer le terminal : ",
        "No command specified.": "Aucune commande spécifiée.",
        "Execute action?": "Exécuter l'action ?",
        "The command will be run in an external terminal.\n\nCommand:\n": "La commande sera exécutée dans un terminal externe.\n\nCommande :\n",
        "Cancel": "Annuler",
        "Execute": "Exécuter",
        "Info": "Info",
        "Dark (click for Light)": "Sombre (cliquer pour Clair)",
        "Light (click for System)": "Clair (cliquer pour Système)",
        "System (click for Dark)": "Système (cliquer pour Sombre)",
        "Quit": "Quitter",
        "Nothing to copy.": "Rien à copier.",
        "Copied to clipboard.": "Copié dans le presse-papiers.",
        "Step 1 · Scanner Basics (SANE)": "Étape 1 · Bases du scanner (SANE)",
        "Installs SANE backends – the foundation for scanner support.": "Installe 'sane' et 'libsane' – la base pour le support des scanners.",
        "Step 2 · User Permissions": "Étape 2 · Permissions utilisateur",
        "Adds your user to the 'scanner' group so devices can be used without root.": "Ajoute votre utilisateur au groupe 'scanner' pour utiliser les appareils sans root.",
        "Grant Permissions": "Accorder les permissions",
        "Step 3 · Install Scanning Apps": "Étape 3 · Installer des apps de scan",
        "Choose an app. Simple Scan is usually the best default choice.": "Choisissez une application. Simple Scan est généralement le meilleur choix.",
        "Simple & fast.": "Simple et rapide.",
        "Advanced / classic.": "Avancé / classique.",
        "KDE tool.": "Outil KDE.",
        "Multipage PDFs.": "PDF multipages.",
        "Step 4 · Manufacturer Drivers (Optional)": "Étape 4 · Pilotes constructeur (Optionnel)",
        "Only needed if your device isn't detected properly by SANE.": "Nécessaire uniquement si votre appareil n'est pas détecté par SANE.",
        "Show Manufacturer Drivers": "Afficher les pilotes constructeur",
        "Legacy for older models.": "Legacy pour les anciens modèles.",
        "Newer models.": "Nouveaux modèles.",
        "Status / Log": "Statut / Journal",
        "Clear": "Effacer",
        "Copy": "Copier",
        "Install": "Installer",
        "Uninstall": "Désinstaller",
        "No packages specified.": "Aucun paquet spécifié.",
        "Scanner Assistant ready.": "Assistant Scanner prêt.",
        "Designed by Pinguin-TV · Void-Gemeni Project": "Conçu par Pinguin-TV · Projet Void-Gemeni",
        "Note: 'pkexec' not found. You might need to add 'sudo' to commands manually.": "Note : 'pkexec' introuvable. Ajoutez 'sudo' manuellement si nécessaire.",
        "Brother / AirScan": "Brother / AirScan",
        "Essential for modern network scanners (eSCL/WSD).": "Essentiel pour les scanners réseau modernes (eSCL/WSD).",
        "Epson (Epkowa)": "Epson (Epkowa)",
        "External backend (often needed for specific Epson models).": "Backend externe (souvent nécessaire pour les modèles Epson spécifiques).",
        "IPP-over-USB": "IPP-over-USB",
        "Allows driverless scanning via USB (modern HP/Brother/Canon).": "Permet de scanner sans pilote via USB (HP/Brother/Canon modernes).",
        "Scan for Devices": "Chercher des scanners",
        "List USB": "Lister USB",
        "Service Status": "État du service",
        "Running": "En cours d'exécution",
        "Stopped / Not installed": "Arrêté / Non installé",
        "Important: Re-login required!": "Important : Reconnexion requise !",
        "You have been added to the 'scanner' group.\nPlease logout and login again for this to take effect.": "Vous avez été ajouté au groupe 'scanner'.\nVeuillez vous déconnecter et vous reconnecter pour que cela prenne effet.",
        "Searching...": "Recherche...",
    }
}

def get_translation_func():
    # Fix for Python 3.12+ deprecated locale.getdefaultlocale
    try:
        env_lang = os.environ.get('LANG', 'en')
        if env_lang:
            lang = env_lang.split('_')[0]
        else:
            lang = "en"
    except Exception:
        lang = "en"

    # Fallback to English if language not found
    if lang not in TRANSLATIONS:
        lang = "en"

    def _(text):
        if lang in TRANSLATIONS and text in TRANSLATIONS[lang]:
            return TRANSLATIONS[lang][text]
        return text
    return _

_ = get_translation_func()

APP_TITLE = _("Void Scanner Assistant")
APP_SUBTITLE = _("SANE · Apps · Permissions · Drivers")

# ---------------------------
# Utilities
# ---------------------------
def which(prog: str):
    for p in os.environ.get("PATH", "").split(os.pathsep):
        f = os.path.join(p, prog)
        if os.path.isfile(f) and os.access(f, os.X_OK):
            return f
    return None

def need_root_wrap(cmd: str) -> str:
    cmd = cmd.strip()
    needs = ("xbps-install", "xbps-remove", "usermod", "ln -s", "ln -sf", "sv ")
    if os.geteuid() == 0 or cmd.startswith(("sudo ", "pkexec ")):
        return cmd
    if any(cmd.startswith(n) for n in needs) and which("pkexec"):
        return "pkexec " + cmd
    return cmd

def set_clipboard_text(text: str):
    Gdk.Display.get_default().get_clipboard().set_text(text)

def check_process_running(process_name: str) -> bool:
    if not which("pgrep"):
        return False
    try:
        subprocess.check_call(["pgrep", "-x", process_name], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
        return True
    except subprocess.CalledProcessError:
        return False
    except Exception:
        return False

def run_capture(cmd):
    try:
        return subprocess.check_output(cmd, stderr=subprocess.STDOUT, text=True, timeout=15)
    except subprocess.TimeoutExpired:
        return _("Command takes too long.")
    except subprocess.CalledProcessError as e:
        return e.output
    except FileNotFoundError:
        return f"Command not found: {cmd[0]}"
    except Exception as e:
        return f"Error: {e}"

def load_theme_pref() -> str:
    try:
        if not CFG_FILE.exists():
            return "system"
        cfg = configparser.ConfigParser()
        cfg.read(CFG_FILE, encoding="utf-8")
        return cfg.get("ui", "theme", fallback="system").strip().lower()
    except Exception:
        return "system"

def save_theme_pref(value: str):
    try:
        CFG_DIR.mkdir(parents=True, exist_ok=True)
        cfg = configparser.ConfigParser()
        cfg["ui"] = {"theme": value}
        with CFG_FILE.open("w", encoding="utf-8") as f:
            cfg.write(f)
    except Exception:
        pass

# ---------------------------
# Externes Terminal
# ---------------------------
TERMINAL_CANDIDATES = [
    ("xfce4-terminal", ["xfce4-terminal", "-e"]),
    ("kitty", ["kitty"]),
    ("konsole", ["konsole", "-e"]),
    ("gnome-terminal", ["gnome-terminal", "--"]),
    ("alacritty", ["alacritty", "-e"]),
    ("tilix", ["tilix", "-e"]),
    ("qterminal", ["qterminal", "-e"]),
    ("xterm", ["xterm", "-e"]),
    ("foot", ["foot", "-e"]),
    ("wezterm", ["wezterm", "start", "--"]),
]

def find_terminal():
    for name, argv in TERMINAL_CANDIDATES:
        if which(name):
            return name, argv
    return None, None

def build_terminal_exec(cmdline: str):
    msg = _("Done. Press any key to close...")
    hold = f"echo; read -n 1 -s -r -p '{msg}' </dev/tty"
    term_name, argv = find_terminal()
    if not term_name:
        return None
    if term_name in ("kitty", "gnome-terminal", "wezterm"):
        return argv + ["bash", "-lc", cmdline + f"; {hold}"]
    full = f"bash -lc {shlex.quote(cmdline + f'; {hold}')}"
    return argv + [full]


# ---------------------------
# App
# ---------------------------
BaseApp = Adw.Application if USE_ADW else Gtk.Application

class ScannerHelperApp(BaseApp):
    def __init__(self):
        super().__init__(application_id=APP_ID)

        self.win = None
        self.toast_overlay = None

        self.theme_pref = load_theme_pref()
        self.btn_theme = None

        self.log_view = None
        
        self.lbl_svc_avahi = None
        self.lbl_svc_ipp = None

        self._apply_css()

    # ---------- Icon fallback ----------
    def icon_or_fallback(self, name: str, fallback: str = "applications-system-symbolic") -> str:
        try:
            theme = Gtk.IconTheme.get_for_display(Gdk.Display.get_default())
            return name if theme.has_icon(name) else fallback
        except Exception:
            return fallback

    # ---------- Styling ----------
    def _apply_css(self):
        css = """
        .card {
            border-radius: 18px;
            padding: 14px;
            background: alpha(@theme_bg_color, 0.55);
            border: 1px solid alpha(@borders, 0.35);
        }
        .section-title { font-weight: 800; font-size: 16px; letter-spacing: 0.2px; }
        .muted { opacity: 0.85; }
        .status-ok { color: #2ec27e; font-weight: bold; }
        .status-err { color: #e01b24; font-weight: bold; }
        textview, entry { border-radius: 12px; }
        button.pill { border-radius: 999px; padding: 6px 10px; }

        button.suggested-action {
            background-image: none;
            background-color: #33BDB4;
            color: #ffffff;
        }
        button.suggested-action:hover { background-color: #2AA79F; }
        button.suggested-action:active { background-color: #23968F; }
        """
        provider = Gtk.CssProvider()
        provider.load_from_data(css)
        Gtk.StyleContext.add_provider_for_display(
            Gdk.Display.get_default(),
            provider,
            Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION,
        )

    # ---------- Theme ----------
    def _update_theme_button(self):
        if not self.btn_theme:
            return
        pref = (self.theme_pref or "system").lower()
        if pref == "dark":
            self.btn_theme.set_icon_name("weather-clear-night-symbolic")
            self.btn_theme.set_tooltip_text(_("Dark (click for Light)"))
        elif pref == "light":
            self.btn_theme.set_icon_name("weather-clear-symbolic")
            self.btn_theme.set_tooltip_text(_("Light (click for System)"))
        else:
            self.btn_theme.set_icon_name("preferences-desktop-theme-symbolic")
            self.btn_theme.set_tooltip_text(_("System (click for Dark)"))

    def apply_theme(self):
        pref = (self.theme_pref or "system").lower()
        if USE_ADW:
            sm = Adw.StyleManager.get_default()
            if pref == "dark":
                sm.set_color_scheme(Adw.ColorScheme.FORCE_DARK)
            elif pref == "light":
                sm.set_color_scheme(Adw.ColorScheme.FORCE_LIGHT)
            else:
                sm.set_color_scheme(Adw.ColorScheme.DEFAULT)
        self._update_theme_button()

    def toggle_theme(self, *_):
        cur = (self.theme_pref or "system").lower()
        nxt = {"system": "dark", "dark": "light", "light": "system"}.get(cur, "system")
        self.theme_pref = nxt
        save_theme_pref(self.theme_pref)
        self.apply_theme()
        self.toast(f"Theme: {self.theme_pref}")

    # ---------- Helpers ----------
    def _section_title(self, text: str) -> Gtk.Label:
        lbl = Gtk.Label(xalign=0.0, label=text)
        lbl.add_css_class("section-title")
        return lbl

    def append_log(self, text: str):
        buf = self.log_view.get_buffer()
        buf.insert(buf.get_end_iter(), text)
        end = buf.get_end_iter()
        mark = buf.create_mark(None, end, True)
        self.log_view.scroll_to_mark(mark, 0.0, True, 0.0, 1.0)

    def toast(self, text: str):
        if USE_ADW and self.toast_overlay:
            self.toast_overlay.add_toast(Adw.Toast.new(text))
        else:
            self.append_log(f"[INFO] {text}\n")

    def _copy_text(self, text: str):
        text = (text or "").strip()
        if not text:
            self.toast(_("Nothing to copy."))
            return
        set_clipboard_text(text)
        self.toast(_("Copied to clipboard."))
        
    def check_services_ui(self):
        is_avahi = check_process_running("avahi-daemon")
        is_ipp = check_process_running("ipp-usb")
        
        def set_status(lbl, is_running):
            if is_running:
                lbl.set_label("● " + _("Running"))
                lbl.remove_css_class("status-err")
                lbl.add_css_class("status-ok")
            else:
                lbl.set_label("● " + _("Stopped / Not installed"))
                lbl.remove_css_class("status-ok")
                lbl.add_css_class("status-err")

        if self.lbl_svc_avahi:
            set_status(self.lbl_svc_avahi, is_avahi)
        if self.lbl_svc_ipp:
            set_status(self.lbl_svc_ipp, is_ipp)

    # ---------- External run ----------
    def run_in_external_terminal(self, cmdline: str):
        term_cmd = build_terminal_exec(cmdline)
        if not term_cmd:
            self.append_log(f"\n[ERROR] {_('No terminal found. Install xfce4-terminal / kitty / konsole.')}\n")
            self.toast(_("No terminal found."))
            return
        try:
            subprocess.Popen(term_cmd)
            self.append_log(f"\n[RUN] {cmdline}\n")
            self.toast(_("Terminal started."))
        except Exception as e:
            self.append_log(f"\n[ERROR] {_('Could not start terminal: ')}{e}\n")
            self.toast(_("Terminal could not be started."))

    def confirm_and_run(self, cmd: str, post_action_cb=None):
        cmd = (cmd or "").strip()
        if not cmd:
            self.toast(_("No command specified."))
            return
        cmd = need_root_wrap(cmd)

        detail = (
            _("The command will be run in an external terminal.\n\nCommand:\n") +
            f"{cmd}"
        )

        if hasattr(Gtk, "AlertDialog"):
            dlg = Gtk.AlertDialog(
                message=_("Execute action?"),
                detail=detail,
                buttons=[_("Cancel"), _("Execute")],
                default_button=1,
                cancel_button=0,
                modal=True,
            )

            def _done(dialog, res):
                try:
                    idx = dialog.choose_finish(res)
                except Exception:
                    return
                if idx == 1:
                    self.run_in_external_terminal(cmd)
                    if post_action_cb:
                        post_action_cb()

            dlg.choose(self.win, None, _done)
        else:
            self.run_in_external_terminal(cmd)
            if post_action_cb:
                post_action_cb()

    # ---------------------------
    # UI Build
    # ---------------------------
    def do_activate(self):
        if self.win:
            self.win.present()
            return

        if USE_ADW:
            Adw.init()
            self.win = Adw.ApplicationWindow(application=self)
            self.win.set_title(APP_TITLE)
            # Increased Height
            self.win.set_default_size(980, 900)
            toolbar = self._build_adw_toolbar_view()
            self.win.set_content(toolbar)
        else:
            self.win = Gtk.ApplicationWindow(application=self, title=APP_TITLE)
            # Increased Height
            self.win.set_default_size(980, 900)
            hb = self._build_gtk_headerbar()
            self.win.set_titlebar(hb)
            root = self._build_content_root()
            self.win.set_child(root)

        self.apply_theme()
        self.check_services_ui()
        self.win.present()

    def _build_gtk_headerbar(self):
        hb = Gtk.HeaderBar()
        title = Gtk.Label()
        title.set_markup(f"<span size='large'><b>{APP_TITLE}</b></span>")
        subtitle = Gtk.Label(label=APP_SUBTITLE)
        subtitle.add_css_class("muted")
        titlebox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=2)
        titlebox.append(title)
        titlebox.append(subtitle)
        hb.set_title_widget(titlebox)

        btn_about = Gtk.Button(icon_name=self.icon_or_fallback("help-about-symbolic", "dialog-information-symbolic"))
        btn_about.add_css_class("pill")
        btn_about.set_tooltip_text(_("Info"))
        btn_about.connect("clicked", lambda *_: self.on_info())

        self.btn_theme = Gtk.Button(icon_name=self.icon_or_fallback("preferences-desktop-theme-symbolic"))
        self.btn_theme.add_css_class("pill")
        self.btn_theme.connect("clicked", self.toggle_theme)
        self._update_theme_button()

        btn_quit = Gtk.Button(icon_name=self.icon_or_fallback("application-exit-symbolic", "window-close-symbolic"))
        btn_quit.add_css_class("pill")
        btn_quit.set_tooltip_text(_("Quit"))
        btn_quit.connect("clicked", lambda *_: self.quit())

        hb.pack_start(btn_about)
        hb.pack_end(btn_quit)
        hb.pack_end(self.btn_theme)
        return hb

    def _build_adw_toolbar_view(self):
        toolbar = Adw.ToolbarView()
        header = Adw.HeaderBar()
        title = Gtk.Label()
        title.set_markup(f"<span size='large'><b>{APP_TITLE}</b></span>")
        subtitle = Gtk.Label(label=APP_SUBTITLE)
        subtitle.add_css_class("muted")
        titlebox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=2)
        titlebox.append(title)
        titlebox.append(subtitle)
        header.set_title_widget(titlebox)

        btn_about = Gtk.Button(icon_name=self.icon_or_fallback("help-about-symbolic", "dialog-information-symbolic"))
        btn_about.add_css_class("pill")
        btn_about.set_tooltip_text(_("Info"))
        btn_about.connect("clicked", lambda *_: self.on_info())

        self.btn_theme = Gtk.Button(icon_name=self.icon_or_fallback("preferences-desktop-theme-symbolic"))
        self.btn_theme.add_css_class("pill")
        self.btn_theme.connect("clicked", self.toggle_theme)
        self._update_theme_button()

        btn_quit = Gtk.Button(icon_name=self.icon_or_fallback("application-exit-symbolic", "window-close-symbolic"))
        btn_quit.add_css_class("pill")
        btn_quit.set_tooltip_text(_("Quit"))
        btn_quit.connect("clicked", lambda *_: self.quit())

        header.pack_start(btn_about)
        header.pack_end(btn_quit)
        header.pack_end(self.btn_theme)

        toolbar.add_top_bar(header)
        root = self._build_content_root()
        self.toast_overlay = Adw.ToastOverlay()
        self.toast_overlay.set_child(root)
        toolbar.set_content(self.toast_overlay)
        return toolbar

    def _build_content_root(self):
        scroller = Gtk.ScrolledWindow(hexpand=True, vexpand=True)
        scroller.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)

        outer = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=14)
        outer.set_margin_top(16)
        outer.set_margin_bottom(16)
        outer.set_margin_start(16)
        outer.set_margin_end(16)
        scroller.set_child(outer)

        # Step 1
        card1 = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        card1.add_css_class("card")
        card1.append(self._section_title(_("Step 1 · Scanner Basics (SANE)")))
        
        status_box = Gtk.Box(spacing=20)
        self.lbl_svc_avahi = Gtk.Label(label="Avahi")
        self.lbl_svc_ipp = Gtk.Label(label="ipp-usb")
        
        status_box.append(Gtk.Label(label="<b>Avahi:</b>", use_markup=True))
        status_box.append(self.lbl_svc_avahi)
        status_box.append(Gtk.Label(label="<b>ipp-usb:</b>", use_markup=True))
        status_box.append(self.lbl_svc_ipp)
        
        card1.append(status_box)
        
        base = Gtk.Label(xalign=0.0, wrap=True, label=_("Installs SANE backends – the foundation for scanner support."))
        base.add_css_class("muted")
        card1.append(base)
        card1.append(self._package_action_row(SCANNER_PACKAGES["SANE"]))

        # Step 2
        card2 = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        card2.add_css_class("card")
        card2.append(self._section_title(_("Step 2 · User Permissions")))
        p = Gtk.Label(xalign=0.0, wrap=True)
        p.set_text(_("Adds your user to the 'scanner' group so devices can be used without root."))
        p.add_css_class("muted")

        btn_perm = Gtk.Button(
            label=_("Grant Permissions"),
            icon_name=self.icon_or_fallback("system-users-symbolic", "avatar-default-symbolic"),
        )
        btn_perm.add_css_class("suggested-action")
        btn_perm.set_tooltip_text(_("Run in external terminal"))
        btn_perm.connect("clicked", lambda *_: self.on_grant_permissions_clicked())

        card2.append(p)
        card2.append(btn_perm)

        # Step 3
        card3 = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        card3.add_css_class("card")
        card3.append(self._section_title(_("Step 3 · Install Scanning Apps")))
        h = Gtk.Label(xalign=0.0, wrap=True, label=_("Choose an app. Simple Scan is usually the best default choice."))
        h.add_css_class("muted")
        card3.append(h)

        grid = Gtk.Grid(column_spacing=12, row_spacing=12)
        grid.set_margin_top(6)
        grid.attach(self._software_card("Simple Scan", _("Simple & fast."), SCANNER_PACKAGES["Simple Scan"]), 0, 0, 1, 1)
        grid.attach(self._software_card("XSane", _("Advanced / classic."), SCANNER_PACKAGES["XSane"]), 1, 0, 1, 1)
        grid.attach(self._software_card("Skanlite", _("KDE tool."), SCANNER_PACKAGES["Skanlite"]), 0, 1, 1, 1)
        grid.attach(self._software_card("gscan2pdf", _("Multipage PDFs."), SCANNER_PACKAGES["gscan2pdf"]), 1, 1, 1, 1)
        card3.append(grid)

        # Step 4
        card4 = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        card4.add_css_class("card")
        card4.append(self._section_title(_("Step 4 · Manufacturer Drivers (Optional)")))
        vh = Gtk.Label(xalign=0.0, wrap=True, label=_("Only needed if your device isn't detected properly by SANE."))
        vh.add_css_class("muted")
        card4.append(vh)

        exp = Gtk.Expander(label=_("Show Manufacturer Drivers"))
        exp.set_expanded(True)

        grid2 = Gtk.Grid(column_spacing=12, row_spacing=12)
        grid2.set_margin_top(10)
        grid2.attach(self._software_card("Epson IScan", _("Legacy for older models."), SCANNER_PACKAGES["Epson IScan"]), 0, 0, 1, 1)
        grid2.attach(self._software_card("Epson ImageScan", _("Newer models."), SCANNER_PACKAGES["Epson ImageScan"]), 1, 0, 1, 1)
        
        grid2.attach(self._software_card(_("Epson (Epkowa)"), _("External backend (often needed for specific Epson models)."), SCANNER_PACKAGES["Epson (Epkowa)"]), 0, 1, 1, 1)
        grid2.attach(self._software_card(_("Brother / AirScan"), _("Essential for modern network scanners (eSCL/WSD)."), SCANNER_PACKAGES["Brother (AirScan)"]), 1, 1, 1, 1)
        
        # Feature 4: IPP-USB
        grid2.attach(self._software_card(_("IPP-over-USB"), _("Allows driverless scanning via USB (modern HP/Brother/Canon)."), SCANNER_PACKAGES["IPP-over-USB"]), 0, 2, 2, 1)

        exp.set_child(grid2)
        card4.append(exp)

        # Log & Diagnostics
        card5 = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        card5.add_css_class("card")
        card5.append(self._section_title(_("Status / Log")))
        
        diag_box = Gtk.Box(spacing=10)
        btn_scan_list = Gtk.Button(label=_("Scan for Devices"), icon_name="edit-find-symbolic")
        btn_scan_list.add_css_class("suggested-action")
        # FIXED: Variable name shadowing (replaced *_ with *_args)
        btn_scan_list.connect("clicked", self.on_scan_devices)
        
        btn_lsusb = Gtk.Button(label=_("List USB"), icon_name="drive-removable-media-usb-symbolic")
        # FIXED: Variable name shadowing (replaced *_ with *_args)
        btn_lsusb.connect("clicked", self.on_list_usb)
        
        diag_box.append(btn_scan_list)
        diag_box.append(btn_lsusb)
        card5.append(diag_box)

        self.log_view = Gtk.TextView(editable=False, monospace=True, wrap_mode=Gtk.WrapMode.WORD_CHAR)
        self.log_view.set_vexpand(True)

        log_scroll = Gtk.ScrolledWindow(hexpand=True, vexpand=True)
        log_scroll.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
        log_scroll.set_child(self.log_view)
        log_scroll.set_min_content_height(200)

        btns = Gtk.Box(spacing=10)
        btn_clear = Gtk.Button(
            label=_("Clear"),
            icon_name=self.icon_or_fallback("edit-clear-all-symbolic", "edit-clear-symbolic"),
        )
        btn_clear.add_css_class("pill")
        btn_clear.connect("clicked", lambda *_: self.log_view.get_buffer().set_text(""))

        btn_copy = Gtk.Button(
            label=_("Copy"),
            icon_name=self.icon_or_fallback("edit-copy-symbolic", "edit-copy-symbolic"),
        )
        btn_copy.add_css_class("pill")
        btn_copy.connect(
            "clicked",
            lambda *_: self._copy_text(self.log_view.get_buffer().get_text(*self.log_view.get_buffer().get_bounds(), True)),
        )

        btns.append(btn_clear)
        btns.append(btn_copy)

        card5.append(log_scroll)
        card5.append(btns)

        outer.append(card1)
        outer.append(card2)
        outer.append(card3)
        outer.append(card4)
        outer.append(card5)

        self.append_log(f"[INFO] {_('Scanner Assistant ready.')}\n")
        return scroller

    # ---------------------------
    # UI components
    # ---------------------------
    def _package_action_row(self, packages: list[str]) -> Gtk.Widget:
        row = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)

        install = Gtk.Button(
            label=_("Install"),
            icon_name=self.icon_or_fallback("system-software-install-symbolic", "list-add-symbolic"),
        )
        install.add_css_class("suggested-action")
        install.set_tooltip_text(_("Run in external terminal"))
        install.connect("clicked", lambda *_: self.on_run_package_command(packages, "install"))

        remove = Gtk.Button(
            label=_("Uninstall"),
            icon_name=self.icon_or_fallback("user-trash-symbolic", "edit-delete-symbolic"),
        )
        remove.add_css_class("destructive-action")
        remove.set_tooltip_text(_("Run in external terminal"))
        remove.connect("clicked", lambda *_: self.on_run_package_command(packages, "remove"))

        row.append(install)
        row.append(remove)
        return row

    def _software_card(self, title: str, description: str, packages: list[str]) -> Gtk.Widget:
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)
        box.add_css_class("card")

        t = Gtk.Label(xalign=0.0)
        t.set_markup(f"<b>{title}</b>")

        d = Gtk.Label(xalign=0.0, wrap=True, label=description)
        d.add_css_class("muted")

        box.append(t)
        box.append(d)
        box.append(self._package_action_row(packages))
        return box

    # ---------------------------
    # Actions & Logic
    # ---------------------------
    def on_run_package_command(self, package_list: list[str], action: str):
        if not package_list:
            self.toast(_("No packages specified."))
            return

        if action == "install":
            cmd = f"xbps-install -S {' '.join(package_list)}"
            
            # AUTOMATION: Enable service if ipp-usb is installed
            if "ipp-usb" in package_list:
                cmd += " && ln -sf /etc/sv/ipp-usb /var/service/"
        else:
            cmd = f"xbps-remove -R {' '.join(package_list)}"

        self.confirm_and_run(cmd)

    def on_grant_permissions_clicked(self):
        username = getpass.getuser()
        cmd = f"usermod -aG scanner {username}"
        
        def show_logout_warning():
            msg = _("You have been added to the 'scanner' group.\nPlease logout and login again for this to take effect.")
            if hasattr(Gtk, "AlertDialog"):
                dlg = Gtk.AlertDialog(
                    message=_("Important: Re-login required!"),
                    detail=msg,
                    buttons=["OK"],
                    modal=True
                )
                dlg.show(self.win)
            else:
                self.append_log(f"\n[INFO] {msg}\n")

        self.confirm_and_run(cmd, post_action_cb=show_logout_warning)

    # FIXED: Replaced *_ with *_args to avoid confusing the translation function _()
    def on_scan_devices(self, *_args):
        self.append_log(f"\n[CMD] scanimage -L ({_('Searching...')})\n")
        
        def worker():
            if not which("scanimage"):
                GLib.idle_add(self.append_log, "[ERROR] 'scanimage' not found. Install sane-backends.\n")
                return
            out = run_capture(["scanimage", "-L"])
            GLib.idle_add(self.append_log, f"{out}\n")

        threading.Thread(target=worker, daemon=True).start()

    # FIXED: Replaced *_ with *_args here too for consistency
    def on_list_usb(self, *_args):
        self.append_log("\n[CMD] lsusb\n")
        def worker():
            if not which("lsusb"):
                GLib.idle_add(self.append_log, "[ERROR] 'lsusb' not found. Install usbutils.\n")
                return
            out = run_capture(["lsusb"])
            GLib.idle_add(self.append_log, f"{out}\n")
            
        threading.Thread(target=worker, daemon=True).start()

    def on_info(self):
        detail = (
            _("Designed by Pinguin-TV · Void-Gemeni Project")
        )
        if hasattr(Gtk, "AlertDialog"):
            dlg = Gtk.AlertDialog(message=APP_TITLE, detail=detail, buttons=["OK"], modal=True)
            dlg.show(self.win)
        else:
            self.append_log("[INFO] " + detail.replace("\n", " ") + "\n")


def main():
    if not which("pkexec"):
        print(_("Note: 'pkexec' not found. You might need to add 'sudo' to commands manually."))
    app = ScannerHelperApp()
    app.run(None)


if __name__ == "__main__":
    main()