#!/usr/bin/env bash
# FreeOffice Installer/Uninstaller for Void Linux (tgz-based)
# Usage: install-freeoffice.sh [install|uninstall] [name]
# - Der zweite Parameter (name) wird vom Store übergeben, ist hier aber optional.
# - Startet Installation/Deinstallation im Terminal (interaktiv, falls nötig).
set -euo pipefail

ACTION="${1:-install}"
NAME_ARG="${2:-freeoffice}"
TMP_ROOT="$(mktemp -d -t freeoffice-XXXXXX)"
cleanup() { rm -rf "$TMP_ROOT" 2>/dev/null || true; }
trap cleanup EXIT

need_cmd() {
  command -v "$1" >/dev/null 2>&1 || { echo "Fehlt: $1"; exit 1; }
}

log() { printf '[FreeOffice] %s\n' "$*"; }

detect_arch() {
  local m; m="$(uname -m)"
  case "$m" in
    x86_64|amd64) echo "amd64";;
    # FreeOffice ist offiziell nur x86_64 (amd64). Andere Archs ggf. später ergänzen.
    *) echo "unsupported";;
  esac
}

get_latest_tgz_url() {
  # Wir parsen die offizielle Seite "Installing FreeOffice for Linux" und
  # greifen den direkten .tgz-Link ab (enthält Revisionsnummer).
  # Beispiel-Link laut Seite:
  # https://www.softmaker.net/down/softmaker-freeoffice-2024-1224-amd64.tgz
  # Quelle: FreeOffice „Installing FreeOffice for Linux“ (Abschnitt „Installation on other systems“). :contentReference[oaicite:0]{index=0}
  local arch="$1"
  local page url
  page="$(curl -fsSL 'https://www.freeoffice.com/en/service/tips-and-tricks/linux' || true)"
  url="$(grep -Eo "https://[^\" ]*softmaker-freeoffice-2024-[0-9]+-${arch}\.tgz" <<<"$page" | head -n1 || true)"
  if [[ -z "$url" ]]; then
    # Fallback: versuche eine konservative, bekannte Struktur (Revision kann sich ändern)
    url="https://www.softmaker.net/down/softmaker-freeoffice-2024-1224-${arch}.tgz"
  fi
  echo "$url"
}

run_installer_from_tgz() {
  local tgz="$1"
  local workdir="$TMP_ROOT/unpack"
  mkdir -p "$workdir"
  log "Entpacke Archiv …"
  tar -xvzf "$tgz" -C "$workdir"
  # Install-Skript finden (laut Anleitung heißt es 'installfreeoffice'). :contentReference[oaicite:1]{index=1}
  local inst
  inst="$(fd -a 'installfreeoffice$' "$workdir" 2>/dev/null || true)"
  if [[ -z "$inst" ]]; then
    # fallback: simple find
    inst="$(find "$workdir" -type f -name 'installfreeoffice' | head -n1 || true)"
  fi
  [[ -n "$inst" ]] || { echo "Konnte 'installfreeoffice' im Archiv nicht finden."; exit 1; }
  log "Starte Installer (sudo kann nach Passwort fragen) …"
  sudo bash "$inst"
  log "Installation abgeschlossen."
}

do_install() {
  need_cmd curl
  need_cmd tar
  need_cmd grep
  need_cmd awk

  local arch; arch="$(detect_arch)"
  if [[ "$arch" == "unsupported" ]]; then
    echo "Diese FreeOffice-Variante unterstützt nur x86_64/amd64."
    exit 1
  fi

  local url; url="$(get_latest_tgz_url "$arch")"
  log "Ermittle Download: $url"
  local out="$TMP_ROOT/$(basename "$url")"
  log "Lade herunter …"
  curl -fL --retry 3 -o "$out" "$url"
  [[ -s "$out" ]] || { echo "Download fehlgeschlagen: $url"; exit 1; }

  run_installer_from_tgz "$out"

  # Hinweis: Icons/Dateitypen werden gemäß Hersteller gesetzt; optional rücksetzbar:
  # sudo /usr/share/freeoffice2024/mime/remove_icons.sh  (laut Doku). :contentReference[oaicite:2]{index=2}

  # Desktop-Caches auffrischen (falls vorhanden)
  command -v update-desktop-database >/dev/null 2>&1 && sudo update-desktop-database || true
  command -v gtk-update-icon-cache >/dev/null 2>&1 && \
    for theme in /usr/share/icons/hicolor; do
      [[ -d "$theme" ]] && sudo gtk-update-icon-cache -q "$theme" || true
    done

  log "FreeOffice wurde installiert."
}

find_uninstaller() {
  # Versuche, ein Deinstallationsskript zu entdecken – Namensschemata variieren.
  # Beobachtet wurden u.a.:
  #  - /usr/share/freeoffice2024/uninstallfreeoffice
  #  - /usr/share/freeoffice2024/uninstall_smfreeoffice2024
  #  - ältere: uninstall_smfreeoffice2018 (vgl. Community-Beiträge). :contentReference[oaicite:3]{index=3}
  local cands=()
  cands+=(/usr/share/freeoffice*/uninstallfreeoffice)
  cands+=(/usr/share/freeoffice*/uninstall_smfreeoffice*)
  cands+=(/opt/freeoffice*/uninstallfreeoffice)
  cands+=(/opt/freeoffice*/uninstall_smfreeoffice*)
  for f in "${cands[@]}"; do
    if [[ -x "$f" ]]; then
      echo "$f"
      return 0
    fi
  done
  return 1
}

manual_remove() {
  log "Kein offizielles Uninstall-Skript gefunden – Fallback-Entfernung."
  # Entferne typische Dateien/Verzeichnisse vorsichtig
  sudo rm -f /usr/bin/textmakerfree /usr/bin/planmakerfree /usr/bin/presentationsfree || true
  sudo rm -f /usr/share/applications/textmakerfree.desktop \
             /usr/share/applications/planmakerfree.desktop \
             /usr/share/applications/presentationsfree.desktop || true
  sudo rm -rf /usr/share/freeoffice2024 /usr/share/freeoffice21 /usr/share/freeoffice20* || true
  # Icons (typisch hicolor):
  for base in /usr/share/icons/hicolor; do
    [[ -d "$base" ]] || continue
    sudo find "$base" -type f \( -name 'textmakerfree.*' -o -name 'planmakerfree.*' -o -name 'presentationsfree.*' \) -delete || true
  done
  command -v update-desktop-database >/dev/null 2>&1 && sudo update-desktop-database || true
  command -v gtk-update-icon-cache >/dev/null 2>&1 && \
    for theme in /usr/share/icons/hicolor; do
      [[ -d "$theme" ]] && sudo gtk-update-icon-cache -q "$theme" || true
    done
  log "FreeOffice (so gut wie möglich) entfernt."
}

do_uninstall() {
  if uninst="$(find_uninstaller)"; then
    log "Starte Uninstaller: $uninst"
    sudo bash "$uninst"
    log "Deinstallation abgeschlossen."
  else
    manual_remove
  fi
}

case "$ACTION" in
  install)   do_install   ;;
  uninstall) do_uninstall ;;
  *)
    echo "Unbekannte Aktion: $ACTION"
    echo "Verwendung: $0 [install|uninstall]"
    exit 2
    ;;
esac

