import subprocess
import os
import shutil
import pwd
from pathlib import Path

class ISOBuilder:
    def __init__(self, options):
        self.options = options
        self.work_dir = Path("/var/tmp/void-live-build")
        self.squashfs_root = self.work_dir / "squashfs-root"
        self.iso_dir = self.work_dir / "iso"
        
        # Kernel ermitteln
        try:
            self.kernel_version = subprocess.check_output(["uname", "-r"]).decode().strip()
        except:
            self.kernel_version = "unknown"

        # Host User ermitteln (sicherer Weg für pkexec)
        # Wenn wir mit pkexec laufen, stehen UID in PKEXEC_UID oder SUDO_UID
        try:
            uid = os.environ.get('PKEXEC_UID') or os.environ.get('SUDO_UID')
            if uid:
                self.host_user = pwd.getpwuid(int(uid)).pw_name
                self.host_home = Path(pwd.getpwuid(int(uid)).pw_dir)
            else:
                self.host_user = os.getlogin()
                self.host_home = Path.home()
        except:
            # Fallback falls alles fehlschlägt
            self.host_user = "root"
            self.host_home = Path("/root")
            
        print(f"Host User erkannt: {self.host_user} (Home: {self.host_home})")

    def build(self):
        try:
            yield (0.05, "Bereinige Arbeitsverzeichnis...")
            self._prepare_workspace()
            
            yield (0.10, "Kopiere Basis-System (ohne /home)...")
            self._copy_base_system()
            
            yield (0.25, "Kopiere ausgewählte Benutzerdaten...")
            self._handle_user_data()
            
            yield (0.35, "Wende Overlay und zusätzliche Pfade an...")
            self._apply_overlays()
            
            yield (0.45, "Konfiguriere Benutzer (Useradd)...")
            self._configure_users()
            
            yield (0.55, f"Erstelle Initramfs (Dracut {self.kernel_version})...")
            self._prepare_kernel_initramfs()
            
            yield (0.65, "Konfiguriere Bootloader...")
            self._configure_bootloader()
            
            yield (0.75, "Komprimiere Dateisystem (SquashFS)...")
            self._create_squashfs()
            
            yield (0.95, "Erstelle ISO Image...")
            self._create_iso()
            
            yield (1.0, "Fertig!")
            
        except Exception as e:
            import traceback
            traceback.print_exc()
            raise e

    def _prepare_workspace(self):
        if self.work_dir.exists():
            shutil.rmtree(self.work_dir)
        self.work_dir.mkdir(parents=True)
        self.squashfs_root.mkdir()
        self.iso_dir.mkdir()
        (self.iso_dir / "boot" / "grub").mkdir(parents=True)
        (self.iso_dir / "LiveOS").mkdir(parents=True)

    def _copy_base_system(self):
        # Basis Kopie, schließt Home, Tmp, etc aus
        exclude_dirs = [
            'proc', 'sys', 'dev', 'tmp', 'run', 'mnt', 'media', 
            'lost+found', 'var/tmp', 'var/cache', 'home' # Home wird separat behandelt
        ]
        
        # Script Pfade ausschließen
        exclude_dirs.append(str(self.work_dir).lstrip('/'))
        
        exclude_args = []
        for d in exclude_dirs:
            exclude_args.append(f'--exclude=/{d}/*')
            (self.squashfs_root / d).mkdir(parents=True, exist_ok=True)

        cmd = ["rsync", "-aAX", "--delete"] + exclude_args + ["/", str(self.squashfs_root) + "/"]
        subprocess.run(cmd, check=True)
        
        # Cleanups
        with open(self.squashfs_root / "etc" / "fstab", "w") as f: f.write("# Live System\n")
        with open(self.squashfs_root / "etc" / "hostname", "w") as f: f.write("void-live")
        open(self.squashfs_root / "etc" / "machine-id", "w").close()

    def _handle_user_data(self):
        # Bestimme Ziel-User Namen für Pfade
        target_user = "anon" # Default
        if self.options['user_mode'] == 'keep':
            target_user = self.host_user
        elif self.options['user_mode'] == 'new':
            target_user = self.options['new_user']
        
        # Zielverzeichnis erstellen
        target_home = self.squashfs_root / "home" / target_user
        target_home.mkdir(parents=True, exist_ok=True)
        
        # Fall 1: Full Home (vom Host User)
        if self.options['full_home']:
            print(f"Kopiere komplettes Home von {self.host_home} nach {target_home}")
            # Achtung: --exclude für Caches um Platz zu sparen
            cmd = ["rsync", "-aAX", "--exclude=.cache", f"{self.host_home}/", f"{target_home}/"]
            subprocess.run(cmd, check=True)
            
        # Fall 2: Granular
        else:
            parts = self.options['home_parts']
            # Mapping Checkbox Name -> Ordner Name
            mapping = {
                'Documents': 'Documents', 'Pictures': 'Pictures', 
                'Downloads': 'Downloads', '.config': '.config', '.local': '.local'
            }
            
            for key, folder in mapping.items():
                if parts.get(key):
                    src = self.host_home / folder
                    dest = target_home / folder
                    if src.exists():
                        print(f"Kopiere {src}...")
                        dest.parent.mkdir(parents=True, exist_ok=True)
                        subprocess.run(["rsync", "-aAX", str(src), str(target_home)], check=True)

        # Hinweis: Die eigentliche Ownership wird bei _configure_users gefixt.

    def _apply_overlays(self):
        # 1. Benutzerdefinierte Pfade (einzelne Ordner vom Host)
        for path in self.options['custom_paths']:
            p = Path(path)
            if p.exists():
                # Relativen Pfad zum Root bestimmen
                if p.is_absolute():
                    rel = p.relative_to(p.anchor)
                else:
                    rel = p
                    
                dest = self.squashfs_root / rel
                
                # Wenn es ein Ordner ist, Parent erstellen und rsyncen
                if p.is_dir():
                    dest.parent.mkdir(parents=True, exist_ok=True)
                    subprocess.run(["rsync", "-aAX", str(p), str(dest.parent)], check=True)
                else:
                    # Datei
                    dest.parent.mkdir(parents=True, exist_ok=True)
                    shutil.copy2(p, dest)

        # 2. Overlay Ordner (Komplette Struktur drüberlegen)
        ov_path = self.options.get('overlay_path')
        if ov_path and os.path.exists(ov_path):
            print(f"Wende Overlay an: {ov_path}")
            # rsync merged directories
            cmd = ["rsync", "-aAX", f"{ov_path}/", f"{self.squashfs_root}/"]
            subprocess.run(cmd, check=True)

    def _configure_users(self):
        mode = self.options['user_mode']
        root_fs = str(self.squashfs_root)
        
        if mode == 'default':
            pass 
            
        elif mode == 'new':
            user = self.options['new_user']
            pwd_text = self.options['new_pass']
            
            # User erstellen im Chroot
            check = subprocess.run(f"chroot {root_fs} id -u {user}", shell=True)
            if check.returncode != 0:
                print(f"Erstelle User {user} im Chroot...")
                cmd = f"chroot {root_fs} useradd -m -s /bin/bash -G wheel,audio,video,users {user}"
                subprocess.run(cmd, shell=True, check=True)
            
            # Passwort setzen
            if pwd_text:
                cmd = f"echo '{user}:{pwd_text}' | chroot {root_fs} chpasswd"
                subprocess.run(cmd, shell=True, check=True)
                
            # Home Verzeichnis Rechte korrigieren (rekursiv)
            subprocess.run(f"chroot {root_fs} chown -R {user}:{user} /home/{user}", shell=True)

        elif mode == 'keep':
            pass

        # Sicherstellen dass sudo funktioniert für wheel gruppe
        (self.squashfs_root / "etc" / "sudoers.d").mkdir(exist_ok=True)
        with open(self.squashfs_root / "etc" / "sudoers.d/wheel", "w") as f:
            f.write("%wheel ALL=(ALL) ALL\n")

    def _prepare_kernel_initramfs(self):
        boot_path = Path("/boot")
        vmlinuz_src = boot_path / f"vmlinuz-{self.kernel_version}"
        if not vmlinuz_src.exists():
            kernels = list(boot_path.glob("vmlinuz-*"))
            if kernels:
                vmlinuz_src = kernels[0]
                self.kernel_version = vmlinuz_src.name.replace("vmlinuz-", "")
            else: raise FileNotFoundError("Kein Kernel gefunden!")

        shutil.copy2(vmlinuz_src, self.iso_dir / "boot" / "vmlinuz")
        
        initramfs_dest = self.iso_dir / "boot" / "initramfs.img"
        dracut_cmd = [
            "dracut", "--verbose", "--no-hostonly", "--no-hostonly-cmdline",
            "--add", "dmsquash-live", "--force",
            str(initramfs_dest), self.kernel_version
        ]
        subprocess.run(dracut_cmd, check=True)

    def _configure_bootloader(self):
        grub_cfg = self.iso_dir / "boot" / "grub" / "grub.cfg"
        label = "VOID_LIVE"
        config = f"""
set default=0
set timeout=5
insmod efi_gop
insmod font
if loadfont /boot/grub/fonts/unicode.pf2 ; then
    insmod gfxterm
    set gfxmode=auto
    terminal_output gfxterm
fi
menuentry "Void Custom Live" {{
    linux /boot/vmlinuz root=live:CDLABEL={label} rd.live.image ro quiet splash
    initrd /boot/initramfs.img
}}
menuentry "RAM Mode" {{
    linux /boot/vmlinuz root=live:CDLABEL={label} rd.live.image rd.live.ram ro
    initrd /boot/initramfs.img
}}
"""
        with open(grub_cfg, "w") as f: f.write(config)

    def _create_squashfs(self):
        comp = ['-comp', 'xz', '-Xbcj', 'x86'] # Immer High für Custom
        squash_dest = self.iso_dir / "LiveOS" / "squashfs.img"
        cmd = ["mksquashfs", str(self.squashfs_root), str(squash_dest), "-noappend"] + comp
        subprocess.run(cmd, check=True)

    def _create_iso(self):
        if not shutil.which("grub-mkrescue"):
            raise RuntimeError("grub-mkrescue fehlt (Paket: grub, grub-i386-efi, grub-x86_64-efi, xorriso)")
        
        cmd = [
            "grub-mkrescue", "-o", self.options['output'],
            "-volid", "VOID_LIVE",
            "--modules=part_gpt part_msdos",
            str(self.iso_dir)
        ]
        subprocess.run(cmd, check=True)