import subprocess
import os
import shutil
import pwd
import glob
from pathlib import Path

class ISOBuilder:
    def __init__(self, options):
        self.options = options
        self.work_dir = Path("/var/tmp/void-live-build")
        self.squashfs_root = self.work_dir / "squashfs-root"
        self.iso_dir = self.work_dir / "iso"
        
        try:
            self.kernel_version = subprocess.check_output(["uname", "-r"]).decode().strip()
        except:
            self.kernel_version = "unknown"

        # Host User Ermittlung
        try:
            uid = os.environ.get('PKEXEC_UID') or os.environ.get('SUDO_UID')
            if uid:
                self.host_user = pwd.getpwuid(int(uid)).pw_name
                self.host_home = Path(pwd.getpwuid(int(uid)).pw_dir)
                # Host-UID/GID speichern für späteren Gebrauch
                pw_info = pwd.getpwuid(int(uid))
                self.host_uid = pw_info.pw_uid
                self.host_gid = pw_info.pw_gid
            else:
                self.host_user = os.getlogin()
                self.host_home = Path.home()
                self.host_uid = os.getuid()
                self.host_gid = os.getgid()
        except:
            self.host_user = "root"
            self.host_home = Path("/root")
            self.host_uid = 0
            self.host_gid = 0
            
    def build(self):
        try:
            yield (0.05, "Bereinige Arbeitsverzeichnis...")
            self._prepare_workspace()
            
            yield (0.10, "Kopiere Basis-System (ohne /home)...")
            self._copy_base_system()
            
            yield (0.25, "Kopiere ausgewählte Benutzerdaten...")
            self._handle_user_data()
            
            yield (0.35, "Wende Overlay an...")
            self._apply_overlays()
            
            yield (0.40, "Konfiguriere Dienste (Netzwerk, Display Manager)...")
            self._configure_services()
            
            yield (0.45, "Konfiguriere Benutzer...")
            self._configure_users()

            if self.options.get('cleanup'):
                yield (0.50, "Bereinige System (Logs, Cache, Keys)...")
                self._cleanup_system()
            
            yield (0.60, f"Erstelle Initramfs (Dracut {self.kernel_version})...")
            self._prepare_kernel_initramfs()
            
            yield (0.70, "Konfiguriere Bootloader...")
            self._configure_bootloader()
            
            yield (0.80, "Komprimiere Dateisystem (SquashFS)...")
            self._create_squashfs()
            
            yield (0.95, "Erstelle ISO Image...")
            self._create_iso()
            
            yield (1.0, "Fertig!")
            
        except Exception as e:
            import traceback
            traceback.print_exc()
            raise e

    def _prepare_workspace(self):
        if self.work_dir.exists():
            shutil.rmtree(self.work_dir)
        self.work_dir.mkdir(parents=True)
        self.squashfs_root.mkdir()
        self.iso_dir.mkdir()
        (self.iso_dir / "boot" / "grub").mkdir(parents=True)
        (self.iso_dir / "LiveOS").mkdir(parents=True)

    def _copy_base_system(self):
        exclude_dirs = [
            'proc', 'sys', 'dev', 'tmp', 'run', 'mnt', 'media', 
            'lost+found', 'var/tmp', 'var/cache', 'home'
        ]
        exclude_dirs.append(str(self.work_dir).lstrip('/'))
        
        exclude_args = []
        for d in exclude_dirs:
            exclude_args.append(f'--exclude=/{d}/*')
            (self.squashfs_root / d).mkdir(parents=True, exist_ok=True)

        cmd = ["rsync", "-aAX", "--delete"] + exclude_args + ["/", str(self.squashfs_root) + "/"]
        subprocess.run(cmd, check=True)
        
        # Basis-Bereinigung
        with open(self.squashfs_root / "etc" / "fstab", "w") as f: 
            f.write("# Live System\n")
        with open(self.squashfs_root / "etc" / "hostname", "w") as f: 
            f.write("void-live\n")
        
        # WICHTIG: Machine-ID generieren (nicht leer lassen!)
        machine_id = subprocess.check_output(["uuidgen"]).decode().strip().replace('-', '')
        with open(self.squashfs_root / "etc" / "machine-id", "w") as f:
            f.write(machine_id + "\n")

    def _handle_user_data(self):
        target_user = "anon"
        if self.options['user_mode'] == 'keep':
            target_user = self.host_user
        elif self.options['user_mode'] == 'new':
            target_user = self.options['new_user']
        
        target_home = self.squashfs_root / "home" / target_user
        target_home.mkdir(parents=True, exist_ok=True)
        
        if self.options['full_home']:
            cmd = ["rsync", "-aAX", "--exclude=.cache", f"{self.host_home}/", f"{target_home}/"]
            subprocess.run(cmd, check=True)
        else:
            parts = self.options['home_parts']
            mapping = {
                'Documents': 'Documents', 'Pictures': 'Pictures', 
                'Downloads': 'Downloads', '.config': '.config', '.local': '.local'
            }
            for key, folder in mapping.items():
                if parts.get(key):
                    src = self.host_home / folder
                    dest = target_home / folder
                    if src.exists():
                        dest.parent.mkdir(parents=True, exist_ok=True)
                        subprocess.run(["rsync", "-aAX", str(src), str(target_home)], check=True)

    def _apply_overlays(self):
        for path in self.options['custom_paths']:
            p = Path(path)
            if p.exists():
                dest = self.squashfs_root / (p.relative_to(p.anchor) if p.is_absolute() else p)
                if p.is_dir():
                    dest.parent.mkdir(parents=True, exist_ok=True)
                    subprocess.run(["rsync", "-aAX", str(p), str(dest.parent)], check=True)
                else:
                    dest.parent.mkdir(parents=True, exist_ok=True)
                    shutil.copy2(p, dest)

        ov_path = self.options.get('overlay_path')
        if ov_path and os.path.exists(ov_path):
            cmd = ["rsync", "-aAX", f"{ov_path}/", f"{self.squashfs_root}/"]
            subprocess.run(cmd, check=True)

    def _configure_services(self):
        """
        Aktiviert NetworkManager, D-Bus und den gewählten Display Manager
        durch Symlinks in /etc/runit/runsvdir/default
        """
        runit_def = self.squashfs_root / "etc/runit/runsvdir/default"
        sv_dir = self.squashfs_root / "etc/sv"
        
        runit_def.mkdir(parents=True, exist_ok=True)
        
        # 1. Standard-Dienste (NetworkManager + D-Bus)
        # WICHTIG: dhcpcd NICHT aktivieren wenn NetworkManager läuft!
        essentials = ["dbus", "NetworkManager"]
        
        for s in essentials:
            src = sv_dir / s
            dest = runit_def / s
            if src.exists() and not dest.exists():
                print(f"Aktiviere Service: {s}")
                os.symlink(f"/etc/sv/{s}", dest)
        
        # 2. Display Manager Logik
        dm_choice = self.options.get('display_manager', 'auto')
        dm_service = None
        
        if dm_choice == 'none':
            print("Kein Display Manager gewählt (Terminal-Login)")
            return
            
        elif dm_choice == 'auto':
            # Versuche zu raten was auf dem Host läuft oder installiert ist
            candidates = ['lightdm', 'gdm', 'sddm', 'lxdm', 'slim']
            # Prüfe was im runit ordner des Hosts aktiviert ist
            for c in candidates:
                if (Path("/var/service") / c).exists():
                    dm_service = c
                    break
            # Wenn nicht gefunden, nimm den ersten der in /etc/sv liegt
            if not dm_service:
                for c in candidates:
                    if (sv_dir / c).exists():
                        dm_service = c
                        break
        else:
            # Explizite Auswahl
            dm_service = dm_choice
            
        # 3. DM aktivieren
        if dm_service and (sv_dir / dm_service).exists():
            print(f"Aktiviere Display Manager: {dm_service}")
            dest = runit_def / dm_service
            if not dest.exists():
                os.symlink(f"/etc/sv/{dm_service}", dest)
        elif dm_choice != 'auto':
            print(f"WARNUNG: Gewählter DM '{dm_service}' ist nicht im Image installiert!")

    def _cleanup_system(self):
        """
        Löscht temporäre Dateien, Logs und sensitive Daten vor dem Packen
        """
        root = self.squashfs_root
        print("Starte System-Bereinigung...")
        
        patterns = [
            "var/cache/xbps/*",
            "var/log/*",
            "var/log/journal/*",
            "root/.bash_history",
            "home/*/.bash_history",
            "home/*/.cache/*",
            "etc/ssh/ssh_host_*",
            "etc/NetworkManager/system-connections/*",
            "tmp/*",
            "var/tmp/*"
        ]
        
        for pat in patterns:
            full_pat = str(root / pat)
            matches = glob.glob(full_pat)
            for m in matches:
                try:
                    p = Path(m)
                    if p.name == "." or p.name == "..": continue
                    
                    if p.is_dir():
                        shutil.rmtree(p)
                    else:
                        p.unlink()
                except Exception as e:
                    print(f"Fehler beim Löschen von {m}: {e}")
        
        # WICHTIG: Machine-ID nicht löschen, aber dbus-machine-id entfernen
        dbus_id = root / "var/lib/dbus/machine-id"
        if dbus_id.exists():
            dbus_id.unlink()
        # Symlink erstellen damit D-Bus /etc/machine-id nutzt
        os.symlink("/etc/machine-id", dbus_id)
        
        # Leere Dummy-Dateien anlegen
        open(root / "var/log/wtmp", 'w').close()
        open(root / "var/log/btmp", 'w').close()

    def _configure_users(self):
        mode = self.options['user_mode']
        root_fs = str(self.squashfs_root)

        print(f"Konfiguriere Benutzer (Modus: {mode})...")

        # ------------------------------------------------
        # 1. Root Fallback (immer!)
        # ------------------------------------------------
        print("Setze Root-Passwort auf 'voidlinux'...")
        subprocess.run(
            ["chroot", root_fs, "chpasswd"],
            input=b"root:voidlinux\n",
            check=True
        )

        # ------------------------------------------------
        # 2. Live-Standard Benutzer (immer vorhanden)
        # ------------------------------------------------
        live_user = "live"
        live_pass = "live"

        check = subprocess.run(
            ["chroot", root_fs, "id", "-u", live_user],
            stdout=subprocess.DEVNULL,
            stderr=subprocess.DEVNULL
        )

        if check.returncode != 0:
            print(f"Erstelle Standard Live-Benutzer '{live_user}'...")
            subprocess.run([
                "chroot", root_fs, "useradd",
                "-m",
                "-s", "/bin/bash",
                "-G", "wheel,audio,video,users,network,input,kvm,storage",
                live_user
            ], check=True)

        subprocess.run(
            ["chroot", root_fs, "chpasswd"],
            input=f"{live_user}:{live_pass}\n".encode(),
            check=True
        )

        subprocess.run(
            ["chroot", root_fs, "chown", "-R", f"{live_user}:{live_user}", f"/home/{live_user}"],
            check=False
        )

        # ------------------------------------------------
        # 3. Host-Benutzer kopieren (WICHTIG: Benutzer muss angelegt werden!)
        # ------------------------------------------------
        if mode == 'keep':
            user = self.host_user
            print(f"Kopiere Host-Benutzer '{user}'...")
            
            # Prüfen ob Benutzer existiert
            check = subprocess.run(
                ["chroot", root_fs, "id", "-u", user],
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL
            )
            
            if check.returncode != 0:
                # Benutzer existiert nicht, muss angelegt werden
                print(f"Erstelle Benutzer '{user}' im Live-System...")
                subprocess.run([
                    "chroot", root_fs, "useradd",
                    "-m",
                    "-s", "/bin/bash",
                    "-G", "wheel,audio,video,users,network,input,kvm,storage",
                    user
                ], check=True)
            
            # Passwort auf leer setzen für einfachen Login (oder 'void')
            subprocess.run(
                ["chroot", root_fs, "chpasswd"],
                input=f"{user}:void\n".encode(),
                check=True
            )
            
            # Permissions korrigieren
            subprocess.run(
                ["chroot", root_fs, "chown", "-R", f"{user}:{user}", f"/home/{user}"],
                check=False
            )

        # ------------------------------------------------
        # 4. Neuer Benutzer aus GUI
        # ------------------------------------------------
        elif mode == 'new':
            user = self.options['new_user']
            pwd_text = self.options['new_pass'] or 'void'

            if user:
                print(f"Erstelle zusätzlichen Benutzer '{user}'...")

                check = subprocess.run(
                    ["chroot", root_fs, "id", "-u", user],
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL
                )

                if check.returncode != 0:
                    subprocess.run([
                        "chroot", root_fs, "useradd",
                        "-m",
                        "-s", "/bin/bash",
                        "-G", "wheel,audio,video,users,network,input,kvm,storage",
                        user
                    ], check=True)

                subprocess.run(
                    ["chroot", root_fs, "chpasswd"],
                    input=f"{user}:{pwd_text}\n".encode(),
                    check=True
                )

                subprocess.run(
                    ["chroot", root_fs, "chown", "-R", f"{user}:{user}", f"/home/{user}"],
                    check=False
                )

        # ------------------------------------------------
        # 5. Sudoers (wheel ohne Passwort)
        # ------------------------------------------------
        sudo_file = self.squashfs_root / "etc" / "sudoers.d" / "wheel"
        sudo_file.parent.mkdir(exist_ok=True)

        with open(sudo_file, "w") as f:
            f.write("%wheel ALL=(ALL) NOPASSWD: ALL\n")

        os.chmod(sudo_file, 0o440)
        
        # ------------------------------------------------
        # 6. PAM/Shadow aktualisieren (wichtig für Login!)
        # ------------------------------------------------
        print("Aktualisiere Shadow-Datenbank...")
        subprocess.run(
            ["chroot", root_fs, "pwck", "-s"],
            check=False
        )

    def _prepare_kernel_initramfs(self):
        boot_path = Path("/boot")
        vmlinuz_src = boot_path / f"vmlinuz-{self.kernel_version}"
        if not vmlinuz_src.exists():
            kernels = list(boot_path.glob("vmlinuz-*"))
            if kernels:
                vmlinuz_src = kernels[0]
                self.kernel_version = vmlinuz_src.name.replace("vmlinuz-", "")
            else: raise FileNotFoundError("Kein Kernel gefunden!")

        shutil.copy2(vmlinuz_src, self.iso_dir / "boot" / "vmlinuz")
        
        initramfs_dest = self.iso_dir / "boot" / "initramfs.img"
        dracut_cmd = [
            "dracut", "--verbose", "--no-hostonly", "--no-hostonly-cmdline",
            "--add", "dmsquash-live", "--force",
            str(initramfs_dest), self.kernel_version
        ]
        subprocess.run(dracut_cmd, check=True)

    def _configure_bootloader(self):
        grub_cfg = self.iso_dir / "boot" / "grub" / "grub.cfg"
        
        # Benutzer-definierbarer Name oder Fallback
        iso_name = self.options.get('iso_name', 'Void Custom Live')
        # Volume-Label muss uppercase sein und darf keine Leerzeichen haben
        label = iso_name.upper().replace(' ', '_')[:32]  # Max 32 Zeichen für ISO9660
        
        config = f"""
set default=0
set timeout=5
insmod efi_gop
insmod font
if loadfont /boot/grub/fonts/unicode.pf2 ; then
    insmod gfxterm
    set gfxmode=auto
    terminal_output gfxterm
fi
menuentry "{iso_name}" {{
    linux /boot/vmlinuz root=live:CDLABEL={label} rd.live.image ro quiet splash
    initrd /boot/initramfs.img
}}
menuentry "{iso_name} (RAM Mode)" {{
    linux /boot/vmlinuz root=live:CDLABEL={label} rd.live.image rd.live.ram ro
    initrd /boot/initramfs.img
}}
"""
        with open(grub_cfg, "w") as f: f.write(config)

    def _create_squashfs(self):
        comp = ['-comp', 'xz', '-Xbcj', 'x86'] 
        squash_dest = self.iso_dir / "LiveOS" / "squashfs.img"
        cmd = ["mksquashfs", str(self.squashfs_root), str(squash_dest), "-noappend"] + comp
        subprocess.run(cmd, check=True)

    def _create_iso(self):
        if not shutil.which("grub-mkrescue"):
            raise RuntimeError("grub-mkrescue fehlt (Paket: grub, grub-i386-efi, grub-x86_64-efi, xorriso)")
        
        # Volume-Label wie in _configure_bootloader
        iso_name = self.options.get('iso_name', 'Void Custom Live')
        label = iso_name.upper().replace(' ', '_')[:32]
        
        cmd = [
            "grub-mkrescue", "-o", self.options['output'],
            "-volid", label,
            "--modules=part_gpt part_msdos",
            str(self.iso_dir)
        ]
        subprocess.run(cmd, check=True)
