#!/bin/bash

# --- SYSTEM-UMGEBUNG ---
export LC_ALL=C.UTF-8
export LANG=C.UTF-8

# --- FARB-DEFINITIONEN ---
R='\033[1;31m' G='\033[1;32m' Y='\033[1;33m' B='\033[1;34m' 
M='\033[1;35m' C='\033[1;36m' W='\033[1;37m' NC='\033[0m'

# --- FZF DESIGN ---
export FZF_DEFAULT_OPTS="--color=fg:#f8f8f2,bg:#282a36,hl:#bd93f9 --color=fg+:#f8f8f2,bg+:#44475a,hl+:#bd93f9 --color=info:#ffb86c,prompt:#50fa7b,pointer:#ff79c6 --color=marker:#ff79c6,spinner:#ffb86c,header:#6272a4 --height=95% --layout=reverse --border --margin=1 --padding=1"

# --- PROGRAMM-LISTEN ---
LIST_BURO="LibreOffice-DE|libreoffice libreoffice-i18n-de|xbps
OnlyOffice|org.onlyoffice.desktopeditors|flat
Collabora Office|com.collaboraoffice.Office|flat
WPS Office|com.wps.Office|flat
XMind 8|net.xmind.XMind8|flat
Freeplane|freeplane|xbps
Xournalpp|xournalpp|xbps
Scribus|scribus|xbps
Marknote|org.kde.marknote|flat
Papers (PDF)|org.gnome.Papers|flat"

LIST_GRAFIK="GIMP|gimp|xbps
Inkscape|inkscape|xbps
Blender|blender|xbps
Krita|krita|xbps
Sweet Home 3D|SweetHome3D|xbps
gThumb|gthumb|xbps
Darktable|darktable|xbps
Gradience|com.github.GradienceTeam.Gradience|flat
Dune 3D|io.github.dune3d.dune3d|flat
LazPaint|net.sourceforge.lazpaint.LazPaint|flat
Pinta|com.github.PintaProject.Pinta|flat"

LIST_INTERNET="Firefox-DE|firefox firefox-i18n-de|xbps
Chromium|chromium|xbps
Vivaldi|vivaldi|xbps
Google Chrome|com.google.Chrome|flat
Zen Browser|io.github.zen_browser.zen|flat
Brave Browser|com.brave.Browser|flat
Thunderbird-DE|thunderbird thunderbird-i18n-de|xbps
Evolution Mail|evolution|xbps
Signal Desktop|org.signal.Signal|flat
Element Messenger|element-desktop|xbps
Telegram|org.telegram.desktop|flat
Discord|com.discordapp.Discord|flat"

LIST_MULTIMEDIA="VLC Player|vlc|xbps
Celluloid|celluloid|xbps
Audacity|audacity|xbps
OBS Studio|obs|xbps
Kdenlive|kdenlive|xbps
mpv Player|mpv|xbps
Spotify|com.spotify.Client|flat
Easy Effects|com.github.wwmm.easyeffects|flat
Tauon Music Box|com.github.taiko2k.tauonmb|flat
Recordbox|com.github.vladimiry.recordbox|flat
Gapless|com.github.razaibi.gapless|flat
Simple Wireplumber GUI|io.github.mrvladus.SimpleWireplumberGUI|flat
FFmpeg|ffmpeg|xbps"

LIST_GAMING="Steam|steam|xbps
Lutris|net.lutris.Lutris|flat
Heroic Games|com.heroicgameslauncher.hgl|flat
Bottles|com.usebottles.bottles|flat
RetroArch|retroarch|xbps
RetroDECK|net.retrodeck.retrodeck|flat
HyperPlay|io.hyperplay.HyperPlay|flat
MangoHud|mangohud|xbps
Gamemode|gamemode|xbps"

LIST_TOOLS="Fastfetch|fastfetch|xbps
FileZilla|filezilla|xbps
VS Code|vscode|xbps
BTOP Manager|btop|xbps
Warp (Transfer)|app.drey.Warp|flat
Pika Datensicherung|org.gnome.World.PikaBackup|flat
AppImage Pool|io.github.prateekmedia.appimagepool|flat
ytDownloader|io.github.shiftey.DesktopDownloader|flat
Easyflatpak|it.miamogio.easyflatpak|flat
XTools (Void Utils)|xtools|xbps
Vsv (Service Manager)|vsv|xbps
Micro Editor|micro|xbps
7-Zip|7zip|xbps"

LIST_SYSTEM="Warehouse|io.github.flattool.Warehouse|flat
GPU-Viewer|io.github.arunsivaramanpk.GPUViewer|flat
1Password|com.onepassword.OnePassword|flat
CPU-X|cpu-x|xbps
BleachBit|bleachbit|xbps
Remmina|remmina|xbps"

LIST_TREIBER="NVIDIA DKMS (Full)|nvidia-dkms nvidia-libs-32bit libglvnd-32bit|xbps
AMD Radeon (Mesa 32-bit)|mesa-dri-32bit mesa-vulkan-radeon mesa-vulkan-radeon-32bit vulkan-loader vulkan-loader-32bit libGL-32bit|xbps
Intel Graphics (Mesa 32-bit)|mesa-dri-32bit mesa-vulkan-intel mesa-vulkan-intel-32bit vulkan-loader vulkan-loader-32bit libva-intel-driver|xbps
AMD CPU Microcode|linux-firmware-amd|xbps
Intel CPU Microcode|intel-ucode|xbps"

LIST_DRUCKER="CUPS Basis (Drucker-System)|cups cups-filters system-config-printer|xbps
HP Drucker (HPLIP + GUI)|hplip hplip-gui|xbps
Epson Drucker (ESC/P-R)|epson-inkjet-printer-escpr gutenprint|xbps
Brother Drucker (Brlaser)|brlaser|xbps
Canon Drucker (cnijfilter2 + BJNP)|cnijfilter2 cups-bjnp gutenprint|xbps
Generic Treiber (Foomatic DB)|foomatic-db foomatic-db-engine|xbps"

# --- CACHE FÜR FLATPAK VERSIONEN (Performance-Optimierung) ---
declare -A FLATPAK_VERSION_CACHE

# --- DASHBOARD ---
get_stats() {
    local host=$(hostname); local kernel=$(uname -r)
    local pkgs=$(xbps-query -l | wc -l); local flatpaks=$(flatpak list 2>/dev/null | wc -l)
    local ram=$(free -h | awk '/^Mem:/ {print $3 "/" $2}')
    local updates_xbps=$(xbps-install -un 2>/dev/null | wc -l)
    local updates_flat=$(flatpak remote-ls --updates 2>/dev/null | wc -l)
    
    echo -e "\n  ${C}╭──────────────────────────────────────────╮${NC}"
    echo -e "  ${C}│${NC}  ${W}🌐 https://pinguin-tv.de${NC}             ${C}│${NC}"
    echo -e "  ${C}├──────────────────────────────────────────┤${NC}"
    echo -e "  ${C}│${NC}  ${Y}System:${NC}   Void Linux ($host)        "
    echo -e "  ${C}│${NC}  ${Y}Kernel:${NC}   $kernel                    "
    echo -e "  ${C}│${NC}  ${Y}RAM:${NC}      $ram                       "
    echo -e "  ${C}│${NC}  ${Y}Software:${NC} $pkgs XBPS | $flatpaks Flatpak      "
    if [ $updates_xbps -gt 0 ] || [ $updates_flat -gt 0 ]; then
        echo -e "  ${C}│${NC}  ${R}Updates:${NC}  $updates_xbps XBPS | $updates_flat Flatpak 🔴"
    fi
    echo -e "  ${C}╰──────────────────────────────────────────╯${NC}"
}
export -f get_stats

# --- VORSCHAU FUNKTION ---
show_preview() {
    local line=$(echo "$1" | sed 's/\x1b\[[0-9;]*m//g')
    
    if [[ "$line" == *"Home"* ]] || [[ "$line" == *"UNIVERSAL"* ]] || [[ -z "$line" ]] || [[ "$line" == *"[BACK]"* ]] || [[ "$line" == *"Flatpak Basis"* ]]; then
        get_stats
    elif [[ "$line" == *"Kernel"* ]]; then
        echo -e "\n  ${Y}🐧 KERNEL-VERWALTUNG${NC}\n  Aktiv: $(uname -r)"
    else
        # Robuste Extraktion der Paket-ID (Inhalt der letzten eckigen Klammer)
        local p_id=$(echo "$line" | grep -o '\[[^][]*\]$' | tr -d '[]' | awk '{print $1}')
        
        # Fallback für Suche-Ergebnisse [XBPS] pkgname
        if [[ -z "$p_id" ]] && [[ "$line" == *"[XBPS]"* || "$line" == *"[FLAT]"* ]]; then
            p_id=$(echo "$line" | awk '{print $2}')
        fi

        [[ -z "$p_id" ]] && return

        echo -e "\n  ${M}PAKET-INFOS FÜR:${NC} ${W}$p_id${NC}"
        echo -e "  ${W}──────────────────────────────────────────${NC}"

        if [[ "$line" == *"xbps"* ]] || [[ "$line" == *"[XBPS]"* ]]; then
            local repo_ver=$(xbps-query -p pkgver -R "$p_id" 2>/dev/null | sed "s/^$p_id-//")
            local inst_ver=$(xbps-query -p pkgver "$p_id" 2>/dev/null | sed "s/^$p_id-//")
            local short_desc=$(xbps-query -R -p short_desc "$p_id" 2>/dev/null)
            
            echo -e "  ${G}Quelle:${NC}      XBPS Repository"
            [[ -n "$short_desc" ]] && echo -e "  ${G}Beschreibung:${NC} $short_desc"
            echo -e "  ${G}Aktuell:${NC}     ${Y}${repo_ver:-Verfügbar im Repo}${NC}"
            if [[ -n "$inst_ver" ]]; then
                echo -e "  ${G}Installiert:${NC} ${W}$inst_ver${NC} ${G}[✔]${NC}"
                if [[ "$inst_ver" != "$repo_ver" ]] && [[ -n "$repo_ver" ]]; then
                    echo -e "  ${Y}⚠ Update verfügbar!${NC}"
                fi
            else
                echo -e "  ${R}Status:      Nicht installiert${NC}"
            fi
            
        elif [[ "$line" == *"flat"* ]] || [[ "$line" == *"[FLAT]"* ]]; then
            # Flatpak Info - prüfe ob installiert
            local inst_ver=$(flatpak list --app --columns=application,version 2>/dev/null | grep "^$p_id" | awk '{print $2}')
            local repo_ver=""
            
            # Hole Version aus dem Remote (funktioniert für installierte UND nicht installierte)
            if [[ -n "$inst_ver" ]]; then
                # Wenn installiert, versuche cached zuerst (schneller)
                local info_cached=$(flatpak remote-info --cached flathub "$p_id" 2>/dev/null)
                repo_ver=$(echo "$info_cached" | grep -iE "^\s*(Version|version):" | head -1 | awk -F: '{print $2}' | xargs)
            fi
            
            # Wenn leer (nicht cached oder nicht installiert), hole vom Server
            if [[ -z "$repo_ver" ]]; then
                local info_remote=$(flatpak remote-info flathub "$p_id" 2>/dev/null)
                repo_ver=$(echo "$info_remote" | grep -iE "^\s*(Version|version):" | head -1 | awk -F: '{print $2}' | xargs)
            fi
            
            echo -e "  ${C}Quelle:${NC}      Flathub (Flatpak)"
            if [[ -n "$repo_ver" ]]; then
                echo -e "  ${C}Aktuell:${NC}     ${Y}$repo_ver${NC}"
            else
                echo -e "  ${C}Aktuell:${NC}     ${Y}Flathub${NC}"
            fi
            
            if [[ -n "$inst_ver" ]]; then
                echo -e "  ${C}Installiert:${NC} ${W}$inst_ver${NC} ${G}[✔]${NC}"
                if [[ "$inst_ver" != "$repo_ver" ]] && [[ -n "$repo_ver" ]]; then
                    echo -e "  ${Y}⚠ Update verfügbar!${NC}"
                fi
            else
                echo -e "  ${R}Status:      Nicht installiert${NC}"
            fi
        fi
        echo -e "  ${W}──────────────────────────────────────────${NC}"
    fi
}
export -f show_preview

# --- INSTALLATIONS / DEINSTALLATIONS LOGIK ---
toggle_software() {
    local type=$1; local pkg=$2
    [[ -z "$pkg" ]] && return
    clear
    
    local is_installed=false
    local first_pkg=$(echo "$pkg" | awk '{print $1}')
    if [ "$type" == "xbps" ]; then
        xbps-query -p pkgver "$first_pkg" &>/dev/null && is_installed=true
    else
        flatpak list --app | grep -q "$first_pkg" && is_installed=true
    fi

    if [ "$is_installed" = true ]; then
        echo -e "${R}SOFTWARE-VERWALTUNG: $pkg${NC}"
        echo -e "----------------------------------------------------"
        echo -ne "\nOptionen: ${Y}(d)${NC} Deinstallieren | ${B}(i)${NC} Info | ${G}(Enter)${NC} Abbrechen: "
        read -r action
        case "$action" in
            d)
                if [ "$type" == "xbps" ]; then 
                    sudo xbps-remove -R $pkg
                else 
                    flatpak uninstall "$first_pkg" -y
                fi
                ;;
            i)
                clear
                if [ "$type" == "xbps" ]; then
                    xbps-query "$first_pkg"
                else
                    flatpak info "$first_pkg"
                fi
                echo -ne "\n${G}Enter zum Fortfahren...${NC}"; read
                return
                ;;
            *)
                return
                ;;
        esac
    else
        echo -e "${G}INSTALLATION STARTEN...${NC}"
        if [ "$type" == "xbps" ]; then 
            sudo xbps-install -S $pkg
            [[ "$pkg" == *"cups"* ]] && sudo ln -sf /etc/sv/cupsd /var/service/
        else 
            flatpak install flathub "$pkg" -y
        fi
    fi
    echo -ne "\n${G}Fertig.${NC} Enter..."; read
}

# --- UNIVERSAL SUCHE ---
global_search() {
    clear
    echo -ne "${W}Suche (XBPS & Flatpak): ${NC}"; read -r query
    [[ -z "$query" ]] && return
    
    echo -e "${Y}Suche läuft...${NC}"
    local xbps_res=$(xbps-query -Rs "$query" | awk '{print $2}' | sed 's/-[0-9].*//' | sort -u | awk '{print "[XBPS] " $1}')
    local flat_res=$(flatpak search "$query" --columns=application 2>/dev/null | tail -n +1 | awk '{print "[FLAT] " $1}')
    local results=$(echo -e "[BACK] Zurück\n$xbps_res\n$flat_res" | grep -v "^$")
    
    local sel=$(echo "$results" | SHELL=/bin/bash fzf --ansi --preview 'bash -c "show_preview {}"' --preview-window=right:60%)
    [[ -z "$sel" ]] || [[ "$sel" == *"[BACK]"* ]] && return
    local p_type="xbps"; [[ "$sel" == *"[FLAT]"* ]] && p_type="flat"
    toggle_software "$p_type" "$(echo "$sel" | awk '{print $2}')"
}

# --- KATEGORIE MENÜ ---
category_menu() {
    local cat_name=$1; local data=$2
    local list_processed=""
    while IFS='|' read -r name pkg type; do
        local status=" "
        local first_pkg=$(echo "$pkg" | awk '{print $1}')
        if [ "$type" == "xbps" ]; then
            xbps-query -p pkgver "$first_pkg" &>/dev/null && status="${G}[✔]${NC} "
        else
            flatpak list --app | grep -q "$first_pkg" && status="${G}[✔]${NC} "
        fi
        list_processed+="${status}(${type}) ${name} [${pkg}]\n"
    done <<< "$data"

    local sel=$(echo -e "[BACK] Zurück zum Menü\n$list_processed" | fzf --ansi --header "Kategorie: $cat_name" --reverse --preview 'bash -c "show_preview {}"')
    [[ -z "$sel" ]] || [[ "$sel" == *"[BACK]"* ]] && return
    
    local p_type=$(echo "$sel" | grep -o "(.*)" | tr -d "()")
    local p_id=$(echo "$sel" | grep -o '\[[^][]*\]$' | tr -d '[]')
    toggle_software "$p_type" "$p_id"
}

# --- SETUP FLATPAK ---
flatpak_setup() {
    clear
    echo -e "${Y}Flatpak Setup...${NC}"
    sudo xbps-install -S flatpak && flatpak remote-add --if-not-exists flathub https://flathub.org/repo/flathub.flatpakrepo
    echo -e "${G}Fertig! Bitte System neu starten für vollständige Integration.${NC}"; read
}

get_available_kernels() {
    xbps-query -Rs . \
        | awk '{print $2}' \
        | grep -E '^linux[0-9]+\.[0-9]+' \
        | sed 's/-.*//' \
        | sort -V -u
}
# --- KERNEL ---
kernel_management() {
    local k_choice
    k_choice=$(echo -e "➕ Installieren\n➖ Entfernen\n🔙 Zurück" \
        | fzf --ansi --reverse --header "Aktueller Kernel: $(uname -r)")

    [[ -z "$k_choice" ]] && return

    case "$k_choice" in
        *"Installieren"*)
    		local kernels
    		kernels=$(get_available_kernels)

    		if [[ -z "$kernels" ]]; then
        		echo -e "${R}Keine Kernel im Repository gefunden!${NC}"
        		read
        		return
    		fi

    		local sel_k
    		sel_k=$(echo "$kernels" | fzf --ansi --reverse \
        		--header "Kernel zur Installation wählen")

    		[[ -z "$sel_k" ]] && return

    		sudo xbps-install -S "$sel_k" "${sel_k}-headers"
    		echo -e "\n${Y}Hinweis: Bitte System neu starten.${NC}"
    		read
    		;;
        *"Entfernen"*)
            local current_kernel
            current_kernel=$(uname -r | cut -d'-' -f1)

            local installed
            installed=$(xbps-query -l \
                | grep "^ii linux[0-9]" \
                | awk '{print $2}' \
                | grep -vE "headers|dbg|base|firmware|tools" \
                | sort -V)

            local sel_rem
            sel_rem=$(echo "$installed" | fzf --ansi --reverse \
                --header "⚠ Aktuellen Kernel NICHT entfernen!")

            [[ -z "$sel_rem" ]] && return

            if [[ "$sel_rem" == "$current_kernel" ]]; then
                echo -e "${R}FEHLER: Aktueller Kernel kann nicht entfernt werden!${NC}"
                read
                return
            fi

            sudo xbps-remove -Rf "$sel_rem" "${sel_rem}-headers"
            read
            ;;
        *)
            return
            ;;
    esac
}



# --- BACKUP ERSTELLEN ---
create_package_list() {
    clear
    echo -e "${Y}📋 Paketliste erstellen...${NC}\n"
    local backup_dir="$HOME/vstore-backup-$(date +%Y%m%d-%H%M%S)"
    mkdir -p "$backup_dir"
    
    # XBPS Pakete
    xbps-query -l | awk '{print $2}' | sed 's/-[0-9].*//' > "$backup_dir/xbps-packages.txt"
    echo -e "${G}✔${NC} XBPS Pakete: $backup_dir/xbps-packages.txt"
    
    # Flatpak Pakete
    flatpak list --app --columns=application > "$backup_dir/flatpak-packages.txt" 2>/dev/null
    echo -e "${G}✔${NC} Flatpak Pakete: $backup_dir/flatpak-packages.txt"
    
    echo -e "\n${G}Backup erstellt in: ${W}$backup_dir${NC}"
    echo -ne "\nEnter zum Fortfahren..."; read
}

# --- MAIN LOOP ---
while true; do
    main_menu=$(echo -e "🏠 Home\n🔍 >>> UNIVERSAL SUCHE <<<\n📦 Flatpak Basis-Setup (Install + Repo)\n
📁 Büro\n🎨 Grafik\n🌐 Internet\n🎬 Multimedia\n🛠️ Tools\n⚙️ System\n
🖥️ Treiber Grafikkarten\n🖨️ Treiber Drucker\n
🎮 Gaming\n
📦 Update & Wartung\n🐧 Kernel-Verwaltung\n💾 Paketliste sichern\n
❌ Beenden")
    
    choice=$(echo -e "$main_menu" | SHELL=/bin/bash fzf --ansi --preview 'bash -c "show_preview {}"' --preview-window=right:55%)
    [[ -z "$choice" ]] && exit 0
    
    clean_choice=$(echo "$choice" | sed 's/\x1b\[[0-9;]*m//g' | xargs)
    [[ -z "$clean_choice" ]] && continue

    case "$choice" in
        *"Home"*) clear; get_stats; read ;;
        *"UNIVERSAL SUCHE"*) global_search ;;
        *"Flatpak Basis-Setup"*) flatpak_setup ;;
        *"Grafikkarten"*) category_menu "Treiber Grafikkarten" "$LIST_TREIBER" ;;
    	*"Drucker"*) category_menu "Treiber Drucker" "$LIST_DRUCKER" ;;

    	*"Büro"*) category_menu "Büro" "$LIST_BURO" ;;
    	*"Grafik"*) category_menu "Grafik" "$LIST_GRAFIK" ;;
    	*"Internet"*) category_menu "Internet" "$LIST_INTERNET" ;;
    	*"Multimedia"*) category_menu "Multimedia" "$LIST_MULTIMEDIA" ;;
    	*"Gaming"*) category_menu "Gaming" "$LIST_GAMING" ;;
    	*"Tools"*) category_menu "Tools" "$LIST_TOOLS" ;;
    	*"System"*) category_menu "System" "$LIST_SYSTEM" ;;
        *"Paketliste sichern"*) create_package_list ;;
        *"Update"*) 
            maintenance_menu() {
                local sel=$(echo -e "🔄 System Update (XBPS + Flatpak)\n🔄 Nur XBPS Update\n🔄 Nur Flatpak Update\n🧹 XBPS Cache leeren\n🗑️ XBPS Waisen entfernen\n🐧 Alte Kernel (vkpurge)\n📦 Flatpak Cleanup\n🔙 Zurück" | fzf --ansi --reverse)
                [[ -z "$sel" ]] || [[ "$sel" == *"Zurück"* ]] && return
                clear
                case "$sel" in
                    *"System Update"*) 
                        echo -e "${Y}XBPS Update...${NC}"
                        command -v topgrade &>/dev/null && topgrade || sudo xbps-install -Su
                        echo -e "\n${Y}Flatpak Update...${NC}"
                        flatpak update -y
                        ;;
                    *"Nur XBPS"*) sudo xbps-install -Su ;;
                    *"Nur Flatpak"*) flatpak update -y ;;
                    *"Cache"*) sudo xbps-remove -O && sudo xbps-remove -S ;;
                    *"Waisen"*) sudo xbps-remove -o ;;
                    *"vkpurge"*) sudo vkpurge rm all ;;
                    *"Flatpak"*) flatpak uninstall --unused -y && flatpak repair ;;
                esac
                echo -ne "\n${G}Fertig.${NC} Enter..."; read
            }
            maintenance_menu ;;
        *"Kernel-Verwaltung"*) kernel_management ;;
        *"Beenden"*) 
            clear
            echo -e "${C}Danke für die Nutzung von vStore! 👋${NC}\n"
            exit 0 
            ;;
    esac
done
